/*
 * Asterisk -- An open source telephony toolkit.
 *
 * Copyright (C) 2008, Verbio Technologies S.L.
 *
 * Verbio Technologies <support@verbio.com>
 *
 * See http://www.asterisk.org for more information about
 * the Asterisk project. Please do not directly contact
 * any of the maintainers of this project for assistance;
 * the project provides a web site, mailing lists and IRC
 * channels for your use.
 *
 * This program is free software, distributed under the terms of
 * the GNU General Public License Version 2. See the LICENSE file
 * at the top of the source tree.
 */

/*! \file
 *
 * \brief Verbio Speech Technologies connector
 *
 * \author Daniel Juan <djuan@verbio.com>
 */

#include "asterisk.h"

#include <stdio.h>
#include <stdlib.h>
#include <unistd.h>
#include <string.h>

#include "asterisk/file.h"
#include "asterisk/channel.h"
#include "asterisk/module.h"
#include "asterisk/lock.h"
#include "asterisk/linkedlists.h"
#include "asterisk/cli.h"
#include "asterisk/term.h"
#include "asterisk/options.h"
#include "asterisk/speech.h"

/* Verbio voxlib */
#include <voxlib.h> 	/*We also need to link against libvoxlib.
			  voxlib.h and libvoxlib.so are included in 
			  verbio-clients package.
			*/

#define	VERBIO_CODEC 	MC_LIN16

static const char	*VERBIO_CFG		= "verbio.conf";
static const char	*VERBIO_DEF_HOST	= "127.0.0.1";
static const int 	VERBIO_MAXINDEX 	= 32;
static const int 	VERBIO_MAX_RES_SIZE	= 100;
static const int 	VERBIO_MAX_CVAR_SIZE	= 120;
static const int 	VERBIO_MAX_INT_SIZE	= 10;
static const float 	VERBIO_INIT_SIL 	= 300.0;
static const float 	VERBIO_MAX_SIL 		= 200.0;
static const float 	VERBIO_MAX_REF		= 200.0;
static const float 	VERBIO_MIN_REF		= 5.0;
static const int 	VERBIO_DEF_ABS_TIMEOUT	= 30;
static const float	VERBIO_LOW_FACTOR	= 2.5;
static const float	VERBIO_HIGH_FACTOR	= 4.5;
static const float	VERBIO_FINAL_FACTOR	= 0.8;
static const float	VERBIO_FINAL_HIGH_FACTOR= 2.5;
static const float	VERBIO_MIN_HIGH_THRESH	= 500.0;
static const float	VERBIO_AAM_MIN		= 50.0;
static const float	VERBIO_AAM_MAX		= 200.0;

static const float	VERBIO_MAX_DEV		= 800000;
static const int 	VERBIO_MAX_GRAM_NUM	= 100;

static const char	*base_fname	= "rsv-";
static const char	*exten		= "pcm";

typedef enum {VVAD_INIT_SIL,
		VVAD_MAX_SIL,
		VVAD_MAX_REF,
		VVAD_MIN_REF,
		VVAD_LOW_FACTOR,
		VVAD_HIGH_FACTOR,
		VVAD_FINAL_FACTOR,
		VVAD_FINAL_HIGH_FACTOR,
		VVAD_MIN_HIGH_THRESH,
		VVAD_AAM_MIN,
		VVAD_AAM_MAX} VVAD_PARAM;

/*
TODO:	- Allow keep_synthesized_files change (from dialplan)..done
	- Test multiple channels/grammars...done
	- Grammar management...done
	- Device id's management...done
	- Set default grammar type to abnf?...done
	- Set rec results (fill results structs)...done
	- Sometimes unload vcb fails...done
	- On Asterisk restart (CLI) module does not load ...done
	- ** Global grammars.
	- Read default grammar path from config file...done
	- All (when possible) config reading to SpeechCreate (save to struct)...done
	- Add record option (save what users say)....done
	- Add verbose option (via config file or dialplan function)...done
	- VAD on client side (recognition status)...done
	- Set/review flags and states (see speech.h)...done
	- Assign grammar to result...done
	- Dynamic number of grammars per channel (now,hardcoded to 100).
	- Allow users change asr config,
	  asr lang, _timeouts_, and other stuff
	  via dialplan (see: SPEECH_ENGINE() Function)....done (test it)
	- Error handling...test it
	- Clean code
	- Remove unneeded log messages...done
	- ...
	- Look for more TODOs in the code.

	- [*]Stress test (create sipp scenario)
 	


	We need to manage available channel id's on client side,
	verbio api can not handle this operations.
	
	Verbio API:
		- Grammars are referenced by id (given on vox_prevcbxx)
		- We need to specify a port (uniq dev id).
*/	

typedef struct{
	/* grammar name */
	const char *name;
	/* grammar path*/
	const char *path;
	/* grammar mode (ISOLATED / ABNF)*/ /*TODO: Allow CONNECTED?*/
	int mode;
}gram_info;

struct verbio_speech_info{
	/* verbosity */
	int verbose;
	/* default ASR config*/
	const char *conf;
	/* default ASR lang*/
	const char *lang;
	/* verbio device */
	int device;
	/* verbio rec device*/
	int recdev;
	/* client-side vad device */
	VAD_PARAM *vsddev;
	/* dafault grammar path*/
	const char *gram_path;
	/* gram_info. will allow us to directly reference
	  (using verbio's gram id) a grammar*/
	gram_info grammar[100];/* TODO: dynamic num of grammars */
	/* absolute timeout */
	int abs_timeout;
	/* sample count*/
	unsigned long int sample_count;
	/* sample rate x timeout*/
	unsigned long int fs_x_tout;
	/* record what user says*/
	int keep_rec_files;
	/* rec file */
	const char *rec_file_path;
	/* rec file */
	const char *rec_file_full_path;
	/*************************/
	/****** VAD params *******/
	/*************************/
	int voice_detected;/*voice detection flag*/
	float init_sil;
	float max_sil;
	float max_ref;
	float min_ref;
	
	/**********************************/
	/****** Advanced VAD params *******/
	/**********************************/
	float low_factor;
	float high_factor;
	float final_factor;
	float final_high_factor;
	float min_high_thresh;
	float aa_min;
	float aa_max;
};


/* ********************************* */
/* ************ Helpers ************ */
/* ********************************* */
/*! \brief Helper function. Set verbosity flag*/
static void verbio_set_verbose( struct ast_speech *speech, int v)
{
	struct verbio_speech_info *verbio_info;
	verbio_info = (struct verbio_speech_info *)speech->data;
	if( verbio_info != NULL)
		verbio_info->verbose = v;
	else
		ast_log( LOG_ERROR, "Could not modify verbosity\n");
}

/*! \brief Helper function. Get verbosity flag*/
static int verbio_get_verbose( struct ast_speech *speech)
{
	struct verbio_speech_info *verbio_info;
	verbio_info = (struct verbio_speech_info *)speech->data;
	if( verbio_info != NULL)
		return verbio_info->verbose;
	else
		return -1;
}

/*! \brief Helper function. Get current Verbio device (previously set)*/
static int verbio_get_curr_dev( struct ast_speech *speech)
{
	int dev = -2; /* -1 reserved to reference all channels*/
	struct verbio_speech_info *verbio_info;
	verbio_info = (struct verbio_speech_info *)speech->data;

	if( verbio_info != NULL)
		dev = verbio_info->device;
	else
		ast_log( LOG_ERROR, "Could not get a valid Verbio device\n");

	return dev;
}

/*! \brief Helper function. Set current Verbio recdevice*/
static void verbio_set_recdev( struct ast_speech *speech, int rec)
{
	struct verbio_speech_info *verbio_info;
	verbio_info = (struct verbio_speech_info *)speech->data;
	if( verbio_info != NULL)
		verbio_info->recdev = rec;
	else
		ast_log( LOG_ERROR, "Could not set a valid Verbio rec device\n");
}

/*! \brief Helper function. Get current Verbio recdevice*/
static int verbio_get_recdev( struct ast_speech *speech)
{
	int rd = 0;
	struct verbio_speech_info *verbio_info;
	verbio_info = (struct verbio_speech_info *)speech->data;
	
	if( verbio_info != NULL)
		rd = verbio_info->recdev;
	else
		ast_log( LOG_ERROR, "Could not get a valid Verbio rec device\n");

	return rd;
}

/*! \brief Helper function. Set current Verbio vsddevice (client-side VAD)*/
static void verbio_set_vsddev( struct ast_speech *speech, VAD_PARAM *v)
{
	struct verbio_speech_info *verbio_info;
	verbio_info = (struct verbio_speech_info *)speech->data;

	if( verbio_info != NULL)
		verbio_info->vsddev = v;
	else
		ast_log( LOG_ERROR, "Could not set a valid Verbio vsd device\n");
}

/*! \brief Helper function. Get current Verbio vsddevice (client-side VAD)*/
static VAD_PARAM* verbio_get_vsddev( struct ast_speech *speech)
{
	VAD_PARAM *rd = NULL;
	struct verbio_speech_info *verbio_info;
	verbio_info = (struct verbio_speech_info *)speech->data;

	if( verbio_info != NULL)
		rd = verbio_info->vsddev;
	else
		ast_log( LOG_ERROR, "Could not get a valid Verbio vsd device\n");

	return rd;
}

/*! \brief Helper function. Set ASR configuration*/
static int verbio_set_curr_config( struct ast_speech *speech, const char *config)
{
	struct verbio_speech_info *verbio_info;
	verbio_info = (struct verbio_speech_info *)speech->data;

	if( verbio_info != NULL)
	{
		verbio_info->conf = strdup(config);
		return 0;
	}
	else
	{
		ast_log( LOG_ERROR, "Could not set ASR configuration to %s\n", config);
		return -1;
	}
}

/*! \brief Helper function. Get ASR configuration*/
static const char* verbio_get_curr_config( struct ast_speech *speech)
{
	struct verbio_speech_info *verbio_info;
	verbio_info = (struct verbio_speech_info *)speech->data;

	if( verbio_info != NULL)
		return verbio_info->conf;
	else
	{
		ast_log( LOG_ERROR, "Could not get ASR configuration\n"); 
		return NULL;
	}
}

/*! \brief Helper function. Set ASR language*/
static int verbio_set_curr_lang( struct ast_speech *speech, const char *lang)
{
	struct verbio_speech_info *verbio_info;
	verbio_info = (struct verbio_speech_info *)speech->data;

	if( verbio_info != NULL)
	{
		verbio_info->lang = strdup(lang);
		return 0;
	}
	else
	{
		ast_log( LOG_ERROR, "Could not set ASR language to %s\n", lang);
		return -1;
	}
}

/*! \brief Helper function. Get ASR language*/
static const char* verbio_get_curr_lang( struct ast_speech *speech)
{
	struct verbio_speech_info *verbio_info;
	verbio_info = (struct verbio_speech_info *)speech->data;

	if( verbio_info != NULL)
		return verbio_info->lang;
	else
	{
		ast_log( LOG_ERROR, "Could not get ASR language\n");
		return NULL;
	}
}

#if 0
static int verbio_set_global_grammar( struct ast_speech *speech, const char *grm)
{
	/*TODO*/
}
#endif

/*! \brief Helper function. Set default grammar path*/
static int verbio_set_def_gram_path( struct ast_speech *speech, const char *p)
{
	struct verbio_speech_info *verbio_info;
	verbio_info = (struct verbio_speech_info *)speech->data;

	if( verbio_info != NULL)
	{
		verbio_info->gram_path = strdup( p);
		return 0;
	}
	else
	{
		ast_log( LOG_ERROR, "Could not set default grammar path to %s\n", p);
		return -1;
	}
}

/*! \brief Helper function. Get default grammar path*/
static const char* verbio_get_def_gram_path( struct ast_speech *speech)
{
	struct verbio_speech_info *verbio_info;
	verbio_info = (struct verbio_speech_info *)speech->data;

	if( verbio_info != NULL)
		return verbio_info->gram_path;
	else
	{
		ast_log( LOG_ERROR, "Could not get default grammar path\n");
		return NULL;
	}
}

/*! \brief Helper function. Set VAD (Voice Activity Detector) param*/
static int verbio_set_curr_vad_param( struct ast_speech *speech, VVAD_PARAM p, float v)
{
	struct verbio_speech_info *verbio_info;
	verbio_info = (struct verbio_speech_info *)speech->data;

	if( !verbio_info)
	{
		ast_log( LOG_ERROR, "Setting VAD param %d\n", p);
		return -1;
	}

	if( v < 0)
	{
		ast_log( LOG_ERROR, "Not setting negative VAD param %f\n", v);
		return -1;
	}

	switch( p) {
		case VVAD_INIT_SIL:  
			verbio_info->init_sil = v;
		break;
		case VVAD_MAX_SIL:  
			verbio_info->max_sil = v;
		break;
		case VVAD_MAX_REF:  
			verbio_info->max_ref = v;
		break;
  		case VVAD_MIN_REF:  
			verbio_info->min_ref = v;
		break;
  		case VVAD_LOW_FACTOR:  
			verbio_info->low_factor = v;
		break;
  		case VVAD_HIGH_FACTOR:  
			verbio_info->high_factor = v;
		break;
  		case VVAD_FINAL_FACTOR:  
			verbio_info->final_factor = v;
		break;
  		case VVAD_FINAL_HIGH_FACTOR:  
			verbio_info->final_high_factor = v;
		break;
  		case VVAD_MIN_HIGH_THRESH:  
			verbio_info->min_high_thresh = v;
		break;
  		case VVAD_AAM_MIN:  
			verbio_info->aa_min = v;
		break;
  		case VVAD_AAM_MAX:  
			verbio_info->aa_max = v;
		break;
  	}

	return 0;
}

/*! \brief Helper function. Get VAD (Voice Activity Detector) param*/
static float verbio_get_curr_vad_param( struct ast_speech *speech, VVAD_PARAM p)
{
	struct verbio_speech_info *verbio_info;
	verbio_info = (struct verbio_speech_info *)speech->data;

	if( !verbio_info)
	{
		ast_log( LOG_ERROR, "Getting VAD param %d\n", p);
		return -1;
	}

	switch( p) {
		case VVAD_INIT_SIL:  
			return verbio_info->init_sil;
		break;
		case VVAD_MAX_SIL:  
			return verbio_info->max_sil;
		break;
		case VVAD_MAX_REF:  
			return verbio_info->max_ref;
		break;
  		case VVAD_MIN_REF:  
			return verbio_info->min_ref;
		break;
		case VVAD_LOW_FACTOR:  
			return verbio_info->low_factor;
		break;
  		case VVAD_HIGH_FACTOR:  
			return verbio_info->high_factor;
		break;
  		case VVAD_FINAL_FACTOR:  
			return verbio_info->final_factor;
		break;
  		case VVAD_FINAL_HIGH_FACTOR:  
			return verbio_info->final_high_factor;
		break;
  		case VVAD_MIN_HIGH_THRESH:  
			return verbio_info->min_high_thresh;
		break;
  		case VVAD_AAM_MIN:  
			return verbio_info->aa_min;
		break;
  		case VVAD_AAM_MAX:  
			return verbio_info->aa_max;
		break;
  	}

	return -1;
}

/*! \brief Helper function. Check VAD status*/
static int verbio_get_voice_detected( struct ast_speech *speech)
{
	struct verbio_speech_info *verbio_info;
	verbio_info = (struct verbio_speech_info *)speech->data;

	if( verbio_info != NULL)
		return verbio_info->voice_detected;
	else
	{
		ast_log( LOG_ERROR, "Checking voice detection flag\n");
		return -1;
	}
}

/*! \brief Helper function. Modify VAD status*/
static int verbio_set_voice_detected( struct ast_speech *speech, int v)
{
	struct verbio_speech_info *verbio_info;
	verbio_info = (struct verbio_speech_info *)speech->data;

	if( verbio_info != NULL)
	{
		verbio_info->voice_detected = v;
		return 0;
	}
	else
	{
		ast_log( LOG_ERROR, "Setting voice detection\n");
		return -1;
	}
}

/*! \brief Helper function. Add grammar to speech engine grammar list.*/
static int verbio_add_grammar( struct ast_speech *speech, 
					const char *grammar_name, 
					const char *grammar_path, 
					int grammar_id)
{
	int verbose = verbio_get_verbose( speech);
	struct verbio_speech_info *verbio_info;
	verbio_info = (struct verbio_speech_info *)speech->data;

	if( grammar_id > (VERBIO_MAX_GRAM_NUM -1))
	{
		ast_log( LOG_ERROR, "Max grammar number reached [%d]\n", grammar_id);
		return -1;
	}
	
	if( verbio_info != NULL)
	{	
		verbio_info->grammar[grammar_id].name = strdup(grammar_name);
		verbio_info->grammar[grammar_id].path = strdup(grammar_path);
		
		if( verbose > 0)
		{
			ast_log( LOG_NOTICE, "Added grammar: %s %s id:%d\n", 
								grammar_name,  
								grammar_path,
								grammar_id);
		}
		return 0;
	}
	else
	{
		ast_log( LOG_ERROR, "Could not add grammar to list\n");
		return -1;
	}
	
}

/*! \brief Helper function. Get previously loaded grammar id (needed to reference grammars).*/
static int verbio_get_grammar_id( struct ast_speech *speech, char *name/*, char* path*/)
{
	int verbose = verbio_get_verbose( speech);
	int gid = -2; /* -1 may be reserved (to unload all grammars)*/
	struct verbio_speech_info *verbio_info;
	verbio_info = (struct verbio_speech_info *)speech->data;
	int done = 0;
	int id = 0;
	while( (!done) && (id < VERBIO_MAX_GRAM_NUM))/*TODO: dyn number of grammars?*/
	{
		if( verbio_info->grammar[id].name == NULL)
			break;
		
		if( !strcmp( name, verbio_info->grammar[id].name))
		{
			gid = id;
			done = 1;
			if( verbose > 0)
				ast_log( LOG_NOTICE, "Found grammar %s id: %d\n", name, id);
		}
		++id;
	}

	return gid;
}

/*! \brief Helper function. Get grammar mode (abnf, text).*/
static int verbio_get_grammar_mode( struct ast_speech *speech, const char *grammar_path)
{
	int verbose = verbio_get_verbose( speech);
	int mode = GVX_ABNF;

	/* auto grammar mode (abnf, txt) detection*/
	int last_indx = strlen(grammar_path);

	/* Isolated (word list) grammars must have 'txt' extension*/
	/* TODO: more 'elegant' detection.*/
	if( grammar_path[last_indx -1] == 't')
	{
		if( verbose > 0)
			ast_log( LOG_NOTICE, "ISOLATED mode\n");
		mode = GVX_ISOLATED;
	}
	else
	{
		if( verbose > 0)
			ast_log( LOG_NOTICE, "ABNF mode\n");
	}	
	return mode;
}

/*! \brief Helper function. Remove grammar from list*/
static void verbio_remove_gram( struct ast_speech *speech, int g)
{
	int verbose = verbio_get_verbose( speech);
	struct verbio_speech_info *verbio_info;
	verbio_info = (struct verbio_speech_info *)speech->data;
	
	if( verbio_info != NULL)
	{	
		if( verbio_info->grammar[g].name != NULL)
			verbio_info->grammar[g].name = "";

		if( verbio_info->grammar[g].path != NULL)
			verbio_info->grammar[g].path = "";
		
		if( verbose > 0)
			ast_log( LOG_NOTICE, "Grammar id :%d removed\n", g);
	}
	else
		ast_log( LOG_ERROR, "Could not remove grammar id %d\n", g);
}

/*! \brief Helper function. Increase sample count (processed samples)*/
static void verbio_incr_sample_count( struct ast_speech *speech, int len)
{
	/* If len = -1 --> set sample counter to 0 */
	struct verbio_speech_info *verbio_info;
	verbio_info = (struct verbio_speech_info *)speech->data;

	if( verbio_info != NULL)
	{
		if( len < 0)
			verbio_info->sample_count = 0;
		else
			verbio_info->sample_count += len;
		/*ast_log( LOG_NOTICE, "sample count set to %ld\n", 
					verbio_info->sample_count);*/
	}
	else
		ast_log( LOG_ERROR, "Could not increase sample count\n");
}

/*! \brief Helper function. Get number of processed samples*/
static unsigned long int verbio_get_sample_count( struct ast_speech *speech)
{
	int retval = -1;
	struct verbio_speech_info *verbio_info;
	verbio_info = (struct verbio_speech_info *)speech->data;

	if( verbio_info != NULL)
		retval = verbio_info->sample_count;
	else
		ast_log( LOG_ERROR, "Could not get sample count\n");
	
	return retval;
}

/*! \brief Helper function. Set absolute timeout*/
static int verbio_set_abs_timeout( struct ast_speech *speech, int t)
{
	int verbose = verbio_get_verbose( speech);
	struct verbio_speech_info *verbio_info;
	verbio_info = (struct verbio_speech_info *)speech->data;

	if( verbio_info != NULL)
	{
		verbio_info->abs_timeout = t;
		verbio_info->fs_x_tout = t * 8000 * 2;
		
		if( verbose > 0)
			ast_log( LOG_NOTICE, "Abs timeout set to %d\n", 
						verbio_info->abs_timeout);
		return 0;
	}
	else
	{
		ast_log( LOG_ERROR, "Could not set absolute timeout\n");
		return -1;
	}
}

#if 0   /*Not needed*/
/*! \brief Helper function. Get absolute timeout*/
static int verbio_get_abs_timeout( struct ast_speech *speech)
{
	int retval = -1;
	struct verbio_speech_info *verbio_info;
	verbio_info = (struct verbio_speech_info *)speech->data;

	if( verbio_info != NULL)
		retval = verbio_info->abs_timeout;
	else
		ast_log( LOG_ERROR, "Could not get timeout\n");
	
	return retval;
}
#endif

/*! \brief Helper function. Get timeout in samples*/
static unsigned long int verbio_get_fs_x_tout( struct ast_speech *speech)
{
	int retval = -1;
	struct verbio_speech_info *verbio_info;
	verbio_info = (struct verbio_speech_info *)speech->data;

	if( verbio_info != NULL)
		retval = verbio_info->fs_x_tout;
	else
		ast_log( LOG_ERROR, "Could not get fs x timeout\n");
	
	return retval;
}

/*! \brief Helper function. Sets recognition result (fill results struct)*/
static int verbio_set_asr_result( struct ast_speech *speech)
{	
	int 	verbose = verbio_get_verbose( speech);
	int	dev = 0;
	int 	nind = 0;
	int 	index[VERBIO_MAXINDEX+1];
	float 	score[VERBIO_MAXINDEX+1];
	int 	i 	= 0;
	int	scr	= 0;
	struct 	ast_speech_result *result = NULL;
	
	speech->state = AST_SPEECH_STATE_WAIT;

	/* ----------------------------- */
	/* ----- Get verbio device ----- */
	/* ----------------------------- */
	dev = verbio_get_curr_dev( speech);
	if( dev == -2)
		return -1;
	
	/* -------------------------------------- */
	/* ----- Get result from recognizer ----- */
	/* -------------------------------------- */
	if(( nind = vox_recind( dev, VERBIO_MAXINDEX, index, score, 0)) < 0)
		ast_log( LOG_ERROR, "vox_recind %s\n", ATVOX_ERRMSGP( dev));

	/* nind = number of recognized words */
	if( verbose > 0)
		ast_log( LOG_NOTICE, "Number of recognized words: %d\n", nind);
	
	if( nind == 0) 
	{
		/* No words recognized */
		/* TODO: can we distinguish between noword and noinput?*/
		/* TODO:speaker may also said an invalid (not in gram) word*/
		speech->flags = AST_SPEECH_SPOKE; 
	}
	else 
	{
		int gid = ATVOX_IVCB( dev);
		struct verbio_speech_info *verbio_info;
		const char *gram = NULL;
	
		verbio_info = (struct verbio_speech_info *)speech->data;
		if( verbio_info == NULL)
		{
			ast_log(LOG_ERROR, "Grammar id %d Not found  \n", gid);
			return 0;/*TODO: return -1 ??*/
		}
		gram = verbio_info->grammar[gid].name;
		
		/* First result */
		speech->results = ast_calloc(1, sizeof(*speech->results));
		result = speech->results;
		result->text = strdup(vox_word(dev, index[0]));
		scr = (int)score[0];
		if( scr > 1000) scr = 1000;
		scr = scr * 25;
		if( scr > 1000) scr = 1000;
		result->score = scr;
		result->grammar = strdup( gram);

		/* for each recognized word...*/
		for( i = 1; i < nind; i++) 
		{
			result->next = ast_calloc(1, sizeof(*speech->results));
			result = result->next;

			result->text = strdup(vox_word(dev, index[i]));
			scr = (int)score[0];
	  	if( scr > 1000) scr = 1000;
  		scr = scr * 25;
			if( scr > 1000) scr = 1000;
			result->score = scr;
			result->grammar = strdup(gram);
		}
		speech->flags = AST_SPEECH_HAVE_RESULTS;
	}
	speech->state = AST_SPEECH_STATE_DONE;

	/* reset sample counter */
	verbio_incr_sample_count(speech, -1);

	return 0;
}

/*! \brief Helper function. Modify keep rec files flag*/
static void verbio_set_keep_rec_files( struct ast_speech *speech, int k)
{
	int verbose = verbio_get_verbose( speech);
	struct verbio_speech_info *verbio_info;
	verbio_info = (struct verbio_speech_info *)speech->data;

	if( verbio_info != NULL)
	{
		verbio_info->keep_rec_files = k;
		if( verbose > 0)
			ast_log( LOG_NOTICE, "Keeping rec files\n");
	}
	else
	{
		ast_log( LOG_ERROR, "Could no set keep rec files\n");
	}
}

/*! \brief Helper function. Get keep rec files flag*/
static int verbio_get_keep_rec_files( struct ast_speech *speech)
{
	int retval = -1;
	struct verbio_speech_info *verbio_info;
	verbio_info = (struct verbio_speech_info *)speech->data;

	if( verbio_info != NULL)
		retval = verbio_info->keep_rec_files;
	else
		ast_log( LOG_ERROR, "Could not get keep rec files\n");
	
	return retval;
}

/*! \brief Helper function. Set full path (of recorded file)*/
static int verbio_set_rec_file( struct ast_speech *speech)
{
	int verbose = verbio_get_verbose( speech);
	struct verbio_speech_info *verbio_info;
	verbio_info = (struct verbio_speech_info *)speech->data;
	struct ast_config 	*vcfg = NULL;	
	const char		*vcfg_recorded_files_exten = NULL;
	const char		*vcfg_recorded_files_path = NULL;
	
        vcfg = ast_config_load( VERBIO_CFG);
	
	if( !vcfg) 
	{
		ast_log(LOG_ERROR, "Error opening configuration file %s\n", VERBIO_CFG);
		return -1;
	}

	vcfg_recorded_files_exten = ast_variable_retrieve(vcfg, "debug", "recorded_files_exten");
	vcfg_recorded_files_path = ast_variable_retrieve(vcfg, "debug", "recorded_files_path");

	if( verbio_info != NULL)
	{
		char tmp[20];
		char filename[30];
		char path[PATH_MAX];
		time_t t = time (NULL);
		int dev = verbio_get_curr_dev( speech);
		if( dev < 0)
			goto failed;
		strcpy( filename, base_fname);
		if( snprintf( tmp, 20, "%d", dev) > 20) 
			ast_log( LOG_WARNING, "Filename may be truncated\n"); 
		strcat( filename, tmp);
		strcat( filename, "-");
		if( snprintf( tmp, 20, "%d", (int)t) > 20) 
			ast_log( LOG_WARNING, "Filename may be truncated\n"); 
		strcat( filename, tmp);
		strcat( filename, ".");
		if( vcfg_recorded_files_exten)
			strcat( filename, vcfg_recorded_files_exten);
		else
			strcat( filename, exten);
				
		/*strcpy( path, verbio_info->rec_file_path);*/
		strcpy( path, vcfg_recorded_files_path);
		strcat( path, "/");
		strcat( path, filename);
                
		verbio_info->rec_file_full_path = strdup( path);
		if( verbose > 0)
			ast_log( LOG_NOTICE, "Rec file set to %s\n", path);
	}
	else
	{
		ast_log( LOG_ERROR, "Could no set keep rec files\n");
		return -1;
	}

	return 0;

 failed:
	ast_config_destroy( vcfg);
	return -1;
}

/*! \brief Helper function. Append/write samples to rec file*/
static void verbio_write_samples_to_file( struct ast_speech *speech, void *d, int l)
{
	struct verbio_speech_info *verbio_info;
	verbio_info = (struct verbio_speech_info *)speech->data;

	if( verbio_info != NULL)
	{
		FILE *f = fopen( verbio_info->rec_file_full_path, "ab"); /*TODO: performance??...test it*/
		if( f)
		{
			fwrite( d , 1 , l , f);
			fclose( f);
		}
		else
		{
			ast_log(LOG_ERROR, "Opening rec file %s\n", 
				verbio_info->rec_file_full_path);
		}
	}

#if 0 /*TODO: use asterisk frames?*/

	/*struct ast_frame 	*f = (struct ast_frame*)malloc(sizeof(ast_frame));*/
	struct ast_frame 	*f = NULL; /*ast_frame_header_new();*/
	if (!(f = ast_calloc(1, sizeof(*f))))
		return -1;

/*
	f->mallocd_hdr_len = sizeof(*f);
	f->datalen = 0;
	f->samples = 0;
	f->data =  NULL;
	f->src = "verbio";
	f->offset = 0;
	f->mallocd=0;
	f->delivery.tv_sec = 0;
	f->delivery.tv_usec = 0;

*/
	f->offset = AST_FRIENDLY_OFFSET;
	f->frametype = AST_FRAME_VOICE;
	f->subclass = AST_FORMAT_SLINEAR;
	f->data = data;
	f->samples = len;
	f->datalen = 4;

	if( ast_writestream( s, f)) 
	{
		ast_log(LOG_WARNING, "Problem writing frame\n");
		ast_frfree(f);
	}
	ast_frfree( f);
#endif
}

/* ******************************************** */
/* ************ API implementation ************ */
/* ******************************************** */
/*! \brief Find a speech recognition engine of specified name, if NULL then use the default one */
static int verbio_create( struct ast_speech *speech)
{
	struct verbio_speech_info *verbio_info = NULL;
	int 			dev = 0;
	struct ast_config 	*vcfg = NULL;	
	const char 		*vcfg_default_asr_config = NULL;
	const char 		*vcfg_default_asr_lang = NULL;
	const char 		*vcfg_grammar_path	= NULL;
	const char		*vcfg_recorded_files_path = NULL;
	const char		*vcfg_recorded_files_exten = NULL;
        int 			vcfg_keep_recorded_files = 0;
	int			vcfg_verbose		= 0;
	char 			*spkinfo = NULL;
	char 			*asrconf = NULL;
	
	const char 	*tmp = NULL;
	float 		vcfg_max_sil  		= VERBIO_MAX_SIL;
	float 		vcfg_init_sil 		= VERBIO_INIT_SIL;
	float 		vcfg_max_ref 		= VERBIO_MAX_REF;	
	float 		vcfg_min_ref 		= VERBIO_MIN_REF;
	int 		vcfg_absolute_timeout	= VERBIO_DEF_ABS_TIMEOUT;
	float 		vcfg_low_factor		= VERBIO_LOW_FACTOR;
	float 		vcfg_high_factor	= VERBIO_HIGH_FACTOR;
	float 		vcfg_final_factor	= VERBIO_FINAL_FACTOR;
	float 		vcfg_final_high_factor	= VERBIO_FINAL_HIGH_FACTOR;
	float 		vcfg_min_high_thresh	= VERBIO_MIN_HIGH_THRESH;
	float 		vcfg_aam_min		= VERBIO_AAM_MIN;
	float		vcfg_aam_max		= VERBIO_AAM_MAX;
	
	speech->data = ast_calloc(1, sizeof(*verbio_info));
	verbio_info = (struct verbio_speech_info *)speech->data;
	if( !verbio_info)
		return -1; 
	
	speech->state = AST_SPEECH_STATE_NOT_READY;

	/* *********************************** */
	/* Create and set Verbio device (port) */
	/* *********************************** */
	/* 
	  Create a unique id, in order to use it as verbio device id.
	  device = channel = port
	  Ex: if we want to attend 3 calls using recognition a the same time, we
	      need 3 different channels (each identifyed by it's own device id).
	*/
	
	/*TODO: use some kind of 'unique' id. (libuuid? custom?)*/
	dev = 1 + (int) (VERBIO_MAX_DEV * (rand() / (RAND_MAX + 1.0)));
	/*TODO: check if someone is already using this device id... how?*/

	verbio_info->device = dev;

	/* ************************************************ */
	/* Get default ASR params (readed from config file) */
	/* ************************************************ */
	vcfg = ast_config_load( VERBIO_CFG);
	
	if( !vcfg) 
	{
		ast_log(LOG_ERROR, "Error opening configuration file %s\n", VERBIO_CFG);
		return -1;
	}
	if( !(vcfg_default_asr_config = ast_variable_retrieve(vcfg, "asr", "default_config")))
	{
		ast_log( LOG_ERROR, "ASR 'default_config' option missing in %s\n", VERBIO_CFG);
		goto failed;
	}
	if( !(vcfg_default_asr_lang = ast_variable_retrieve(vcfg, "asr", "default_language"))) 
	{
		ast_log( LOG_ERROR, "ASR 'default_language' option missing in %s\n", VERBIO_CFG);
		goto failed;
	}

	if( !(vcfg_grammar_path = ast_variable_retrieve(vcfg, "asr", "grammar_path"))) 
		ast_log( LOG_WARNING, "Could not read grammar_path option\n");

	if( !( tmp = ast_variable_retrieve(vcfg, "debug", "verbose"))) 
		tmp = "0";
	vcfg_verbose = atoi( tmp);
	if( vcfg_verbose)
	{
		ast_log( LOG_NOTICE, "Verbose enabled in config file.\n");
		verbio_set_verbose( speech, 1);
	}

	if( (tmp = ast_variable_retrieve(vcfg, "debug", "keep_recorded_files"))) 
		vcfg_keep_recorded_files = 1;
	
	vcfg_recorded_files_path = ast_variable_retrieve(vcfg, "debug", "recorded_files_path"); 
				
	vcfg_recorded_files_exten = ast_variable_retrieve(vcfg, "debug", "recorded_files_exten"); 
	  
	if( (tmp = ast_variable_retrieve(vcfg, "asr", "init_sil"))) 
		vcfg_init_sil = atof( tmp);

	if( (tmp = ast_variable_retrieve(vcfg, "asr", "max_sil"))) 
		vcfg_max_sil = atof( tmp);
	
	if( (tmp = ast_variable_retrieve(vcfg, "asr", "max_ref"))) 
		vcfg_max_ref = atof( tmp);
	
	if( (tmp = ast_variable_retrieve(vcfg, "asr", "min_ref"))) 
		vcfg_min_ref = atof( tmp);

	if( (tmp = ast_variable_retrieve(vcfg, "asr", "absolute_timeout"))) 
		vcfg_absolute_timeout = atoi( tmp);

	if( (tmp = ast_variable_retrieve(vcfg, "vad", "low_factor"))) 
		vcfg_low_factor = atof( tmp);

	if( (tmp = ast_variable_retrieve(vcfg, "vad", "high_factor"))) 
		vcfg_high_factor = atof( tmp);

	if( (tmp = ast_variable_retrieve(vcfg, "vad", "final_factor"))) 
		vcfg_final_factor = atof( tmp);

	if( (tmp = ast_variable_retrieve(vcfg, "vad", "final_high_factor"))) 
		vcfg_final_high_factor = atof( tmp);

	if( (tmp = ast_variable_retrieve(vcfg, "vad", "min_high_thresh"))) 
		vcfg_min_high_thresh = atof( tmp);

	if( (tmp = ast_variable_retrieve(vcfg, "vad", "aam_min"))) 
		vcfg_aam_min = atof( tmp);

	if( (tmp = ast_variable_retrieve(vcfg, "vad", "aam_max"))) 
		vcfg_aam_max = atof( tmp);

	vox_getparm(-1, VXGB_TTSSPKINFO, &spkinfo);
	if( spkinfo)
	{
		if( verbio_get_verbose( speech) > 0)
			ast_log(LOG_NOTICE,"Available TTS speakers: %s\n", spkinfo);
	}

	vox_getparm(-1, VXGB_START_CONF, &asrconf);
	if( asrconf)
	{
		if( verbio_get_verbose( speech) > 0)
			ast_log(LOG_NOTICE,"Available ASR configs: %s\n", asrconf);
	}
	
	if( verbio_get_verbose( speech) > 0)
	{
		ast_log(LOG_NOTICE, "---- verbio_create going to set: ----\n");
		ast_log(LOG_NOTICE, "- Verbio Device: %d\n", 
					dev);
		ast_log(LOG_NOTICE, "- ASR config: %s\n", 
					vcfg_default_asr_config);
		ast_log(LOG_NOTICE, "- ASR lang: %s\n", 
					vcfg_default_asr_lang);
		if( vcfg_grammar_path)
			ast_log(LOG_NOTICE, "- Def gram path: %s\n", 
							vcfg_grammar_path);
		else
			ast_log(LOG_WARNING, "- grammar_path not defined. Check verbio.conf\n");
			
		ast_log(LOG_NOTICE, "- init sil: %f\n", 
					vcfg_init_sil);
		ast_log(LOG_NOTICE, "- max sil: %f\n", 
					vcfg_max_sil);
		ast_log(LOG_NOTICE, "- max ref: %f\n", 
					vcfg_max_ref);
		ast_log(LOG_NOTICE, "- min ref: %f\n", 
					vcfg_min_ref);
		ast_log(LOG_NOTICE, "- absolute timeout: %d\n", 
					vcfg_absolute_timeout);
		ast_log(LOG_NOTICE, "- low factor: %f\n", 
					vcfg_low_factor);
		ast_log(LOG_NOTICE, "- high factor: %f\n", 
					vcfg_high_factor);
		ast_log(LOG_NOTICE, "- final factor: %f\n", 
					vcfg_final_factor);
		ast_log(LOG_NOTICE, "- final high factor: %f\n", 
					vcfg_final_high_factor);
		ast_log(LOG_NOTICE, "- min high thresh: %f\n", 
					vcfg_min_high_thresh);
		ast_log(LOG_NOTICE, "- aam min: %f\n", 
					vcfg_aam_min);
		ast_log(LOG_NOTICE, "- aam max: %f\n", 
					vcfg_aam_max);
		if( vcfg_keep_recorded_files)
		{
			if( vcfg_recorded_files_path)	
			{			
				ast_log(LOG_NOTICE, "- Rec files path: %s\n", 
						vcfg_recorded_files_path);
				
				if( vcfg_recorded_files_exten)
					ast_log(LOG_NOTICE, "- Rec file exten: %s\n", vcfg_recorded_files_exten);
				else
					ast_log(LOG_NOTICE, "- Rec file exten: %s\n", exten);
			}
			else
			{
				ast_log(LOG_WARNING, "- recorded_files_path not set. Check verbio.conf\n");
			}
		}  
		ast_log(LOG_NOTICE, "-------------------------------------\n");	
	}

	/* ******************************** */
	/* Set previously readed ASR params */
	/* ******************************** */
	if( verbio_set_curr_config( speech, vcfg_default_asr_config) < 0)
		goto failed;
	
	if( verbio_set_curr_lang( speech, vcfg_default_asr_lang) < 0)
		goto failed;

	if( vcfg_grammar_path) 
		verbio_set_def_gram_path( speech, vcfg_grammar_path);
	
	if( verbio_set_curr_vad_param( speech, VVAD_INIT_SIL, vcfg_init_sil) < 0)
		goto failed;
	
	if( verbio_set_curr_vad_param( speech, VVAD_MAX_SIL, vcfg_max_sil) < 0)
		goto failed;
	
	if( verbio_set_curr_vad_param( speech, VVAD_MAX_REF, vcfg_max_ref) < 0)
		goto failed;
	
	if( verbio_set_curr_vad_param( speech, VVAD_MIN_REF, vcfg_min_ref) < 0)
		goto failed;
	
	if( verbio_set_abs_timeout( speech, vcfg_absolute_timeout) < 0)
		goto failed;
	
	if( verbio_set_curr_vad_param( speech, VVAD_LOW_FACTOR, vcfg_low_factor) < 0)
		goto failed;
	
	if( verbio_set_curr_vad_param( speech, VVAD_HIGH_FACTOR, vcfg_high_factor) < 0)
		goto failed;
	
	if( verbio_set_curr_vad_param( speech, VVAD_FINAL_FACTOR, vcfg_final_factor) < 0)
		goto failed;
	
	if( verbio_set_curr_vad_param( speech, VVAD_FINAL_HIGH_FACTOR, vcfg_final_high_factor) < 0)
		goto failed;
	
	if( verbio_set_curr_vad_param( speech, VVAD_MIN_HIGH_THRESH, vcfg_min_high_thresh) < 0)
		goto failed;
	
	if( verbio_set_curr_vad_param( speech, VVAD_AAM_MIN, vcfg_aam_min) < 0)
		goto failed;
	
	if( verbio_set_curr_vad_param( speech, VVAD_AAM_MAX, vcfg_aam_max) < 0)
		goto failed;

	if( vcfg_keep_recorded_files)
	{
		if( vcfg_recorded_files_path)
			verbio_set_keep_rec_files( speech, 1);
		else
		{
			verbio_set_keep_rec_files( speech, 0);
			ast_log(LOG_WARNING, "recorded_files_path not set\n");	
		}
	}
	else
		verbio_set_keep_rec_files( speech, 0);

	verbio_incr_sample_count( speech, -1); /* -1 => Set sample count to 0*/
	
	ast_config_destroy( vcfg);
	return 0;

 failed:
	ast_config_destroy( vcfg);
	return -1;
}

/*! \brief  Destroy connection to engine. */
static int verbio_destroy( struct ast_speech *speech)
{
	int verbose = verbio_get_verbose( speech);

	/*TODO: free structures (results, grammar info,...)*/

	/* Free dev resources...otherwise we will run out of lines*/
	vox_devclose( verbio_get_curr_dev( speech)); 
	
	if( verbose > 0)
		ast_log(LOG_NOTICE, "verbio_destroy connection closed\n");
	
	return 0;
}

/*! \brief Load a local grammar on a speech structure */
static int verbio_load(struct ast_speech *speech, char *grammar_name, char *grammar)
{
	int 		verbose = verbio_get_verbose( speech);
	const char 	*asr_config = verbio_get_curr_config( speech);
	const char 	*asr_lang = verbio_get_curr_lang( speech);
	int 		dev = verbio_get_curr_dev( speech);
	char 		gram_full_path[MAXPATHLEN];
	int 		nind = 0;
	int 		mode = verbio_get_grammar_mode( speech, 
						(const char*)grammar);
	
	if( !asr_config || !asr_lang || dev == -2)
	{
		ast_log(LOG_ERROR, "Fatal error in verbio_load\n");
		return -1;
	}

	if( verbio_get_grammar_id( speech, grammar_name) != -2)
	{
		ast_log(LOG_WARNING, "Grammar %s already loaded\n", 
							grammar_name);
		return 0;
	}	

	/* <Grammar file> */
	if( grammar[0] != '/')
	{
		/* Full path not provided, try to read from default gram path.
		   (default grammar path readed from verbio.conf)
		*/
		const char* def_grm_path = verbio_get_def_gram_path( speech);
		if( !def_grm_path)
		{
			ast_log(LOG_ERROR, "Default grammar path missing\n");	
			return -1;
		}

		strcpy( gram_full_path, def_grm_path); /*FIXME: ast_copy_string? (if exists, use it)*/
		strcat( gram_full_path, "/");	
		strcat( gram_full_path, grammar);	
	}
	else
	{
		strcpy( gram_full_path, grammar);
	}

	
	if( vox_setparm( dev,VXCH_DEFASRCFG, asr_config) == -1)
	{
		ast_log( LOG_ERROR, "Can not set ASR config to %s [%s]\n", 
							asr_config, 
							ATVOX_ERRMSGP( dev));
		return -1;
	}

	if( vox_setparm( dev,VXCH_DEFASRLANG, asr_lang) == -1)
	{
		ast_log( LOG_ERROR, "Can not set ASR lang to %s [%s]\n", 
							asr_lang, 
							ATVOX_ERRMSGP( dev));
		return -1;
	}

	if( verbose > 0)
		ast_log( LOG_NOTICE, "Preparing grammar %s conf:%s lang:%s\n", 
								grammar, 
								asr_config, 
								asr_lang);

	if( vox_prevcbdev( dev, gram_full_path, mode, &nind, asr_lang) == -1) 
	{
		
		ast_log(LOG_ERROR, "Error preparing grammar [%s line: %d]\n", 
						ATVOX_ERRMSGP( dev), nind);
		ast_log(LOG_ERROR, "- grammar path: %s\n", grammar);
	
		return -1;
	}

	nind = vox_loadvcb( dev, gram_full_path, mode);
	if( nind == -1) 
	{
		ast_log(LOG_ERROR, "Error loading grammar [%s line %d]\n",
						ATVOX_ERRMSGP( dev), nind);
		ast_log(LOG_ERROR, "- grammar path: %s\n", grammar);

		return -1;
	}
	
	if( verbio_add_grammar( speech, grammar_name, gram_full_path, nind) < 0)
	{
		ast_log(LOG_ERROR, "Grammar %s not added to list\n", 
							gram_full_path);
		return -1;
	}

	return 0;
}

/*! \brief Unload a local grammar from a speech structure */
static int verbio_unload(struct ast_speech *speech, char *grammar_name)
{
	int dev = verbio_get_curr_dev( speech);
	int gid = verbio_get_grammar_id( speech, grammar_name);
	
	if( dev == -2)
		return -1;

#if 0
	if( gid == -1)/* TODO allow this? (to unload all grammars)*/
	{
		if( vox_setvcb( dev, NULL, 0))/* Free licences */
			ast_log( LOG_ERROR, "vox_setvcb. [%s]\n", ATVOX_ERRMSGP( dev));
		
		vox_devclose( dev); /* Free dev resources...otherwise we will run out of lines*/
	}
#endif

	if( gid != -2)/*get_grammar_id returns -2 when grammar not found*/
	{
		if( vox_unloadvcb( dev, gid, 0) < 0) 
		{
			ast_log( LOG_ERROR, "Error unloading grammar %s [%s]\n", 
							grammar_name, 
							ATVOX_ERRMSGP( dev));
			/*return -1;*/
		}
		/*remove grammar from list*/
		verbio_remove_gram( speech, gid);
	}
	else
	{
		ast_log( LOG_WARNING, "Grammar %s not found\n", grammar_name);
	}
	return 0;
}

/*! \brief Activate a loaded (either local or global) grammar */
static int verbio_activate(struct ast_speech *speech, char *grammar_name)
{
	/* Activate grammar */
	int dev = verbio_get_curr_dev( speech);
	int gid = verbio_get_grammar_id( speech, grammar_name);
	struct verbio_speech_info *verbio_info;
	const char *gram_path = NULL;
	int mode = GVX_ABNF;

	verbio_info = (struct verbio_speech_info *)speech->data;
	if( verbio_info == NULL)
	{
		ast_log(LOG_ERROR, "Grammar %s Not found  \n", grammar_name);
		return 0;/*TODO: return -1 ??*/
	}

	gram_path = verbio_info->grammar[gid].path;
	mode = verbio_get_grammar_mode( speech, gram_path);

	/*TODO: allow this situation? (repeated grammars)*/
	/*get_grammar_id returns -1 when grammar not found*/
	/*while( gid != -1) 
	{*/
	if( gid == -2)/*get_grammar_id returns -2 when grammar not found*/
	{
		ast_log(LOG_ERROR, "Grammar %s not found\n", grammar_name);
		return 0;/*TODO: return -1 ??*/
	}

	if( vox_activatevcb( dev, gid, mode) == -1) 
	{
		ast_log(LOG_ERROR, "Error activating grammar [%s]\n", 
							ATVOX_ERRMSGP( -1));
		return -1;
	}
	
	/*}*/
	return 0;
}

/*! \brief Deactivate a loaded grammar on a speech structure */
static int verbio_deactivate(struct ast_speech *speech, char *grammar_name)
{
	/* Deactivate grammar */
	int dev = verbio_get_curr_dev( speech);
	int gid = verbio_get_grammar_id( speech, grammar_name);
	struct verbio_speech_info *verbio_info;
	const char *gram_path = NULL;
	int mode = GVX_ABNF;

	verbio_info = (struct verbio_speech_info *)speech->data;
	if( verbio_info == NULL)
	{
		ast_log(LOG_ERROR, "Could not deactivate grammar %s\n", 
								grammar_name);
		return 0;/*TODO: return -1 ??*/
	}

	gram_path = verbio_info->grammar[gid].path;
	if( !gram_path)
		return 0; /*TODO: return -1 ??*/

	mode = verbio_get_grammar_mode( speech, gram_path);
	/*int mode = verbio_get_grammar_mode( grammar_name);*/

	/*deactivate (maybe) repeated grammars*/
	/*while( gid != -2) 
	{*/
	if( gid != -2) /*get_grammar_id returns -2 when grammar not found*/
	{
		if( vox_deactivatevcb( dev, gid, mode) == -1) 
		{
			ast_log(LOG_ERROR, "Error deactivating grammar [%s]\n", 
							ATVOX_ERRMSGP( -1));
			return 0; /*TODO: return -1?*/
		}
	}
	else
	{
		ast_log( LOG_WARNING, "Grammar %s not found\n", grammar_name);
	}
	/*
		gid = verbio_get_grammar_id( speech, grammar_name);
	}
	*/
	return 0;
}

/*! \brief Write in signed linear audio to be recognized */
static int verbio_write(struct ast_speech *speech, void *data, int len)
{
	/* Pass samples to Voice Activity Detector or ASR engine.
	   Depending on voice detection status.
	*/
	if( verbio_get_keep_rec_files( speech) > 0)
		verbio_write_samples_to_file( speech, data, len);

	if( verbio_get_voice_detected( speech) != 1) /*User has not spoken yet*/
	{
		/* 
		  		= VAD Client-side = 
			Please edit /etc/asterisk/verbio.conf (vad section)
			if you want to control vad sensitivity.
		*/
		int status = 0;
		VAD_PARAM *vsddev = verbio_get_vsddev( speech);
		status = vox_vsd_write( vsddev, data, len);
		if( (status != VVX_INIT) && (status != VVX_SILENCE))
		{
			/* Voice detected */
			speech->flags = AST_SPEECH_QUIET;/*Quiet prompt.*/
			verbio_set_voice_detected( speech, 1);
			if( verbio_get_verbose( speech) > 0)
				ast_log(LOG_NOTICE, "Voice detected\n");
		}

		/* VAD (Voice Activity Detection) Server-side*/
		/* Best on client-side (do not bother ASR engine with VAD)*/
		/*vox_getparm( dev, VXCH_VOICEDETECTED, &voice_detected);*/
		/*if( voice_detected == 1)
			ast_stopstream( chan);*/
	}
	else 	/* Speech has been detected. 
		Now, we need to send samples to ASR engine.*/
	{
		int recdev = verbio_get_recdev( speech);
		if( vox_recstr_write( recdev, data, len) != len)
			return verbio_set_asr_result( speech);
	}

	/* Increase processed samples counter*/
	verbio_incr_sample_count( speech, len);

	/* absolute timeout check*/
	unsigned long int sample_count = verbio_get_sample_count( speech);
	unsigned long int fs_x_t = verbio_get_fs_x_tout( speech);
	if( sample_count < 0 || fs_x_t < 0)
		return -1;

	if( sample_count >= fs_x_t)
	{
		ast_log(LOG_WARNING, "Absol timeout reached (%ld)\n", 
								sample_count);
		speech->flags = AST_SPEECH_SPOKE; 
		speech->state = AST_SPEECH_STATE_DONE;
		verbio_incr_sample_count( speech, -1); /*reset sample count*/
	}

	return 0;
}

/*! \brief Signal to the engine that DTMF was received */
static int verbio_dtmf(struct ast_speech *speech, const char *dtmf)
{
	/*TODO: implementation??*/
	ast_log(LOG_NOTICE, "Verbio dtmf not implemented\n");
	return 0;
}

/*! \brief Start speech recognition on a speech structure */
static int verbio_start(struct ast_speech *speech)
{
	struct ast_speech_engine *verbio_engine;
	int recdev = 0; /*recdev inited in this function*/
	VAD_PARAM *vsddev = NULL; /* vsddev inited in this function*/
	float 	init_sil = verbio_get_curr_vad_param( speech, VVAD_INIT_SIL);
	float 	max_sil =verbio_get_curr_vad_param( speech, VVAD_MAX_SIL);;
	VX_RSP 	rsp;
	float low_factor = verbio_get_curr_vad_param( speech, VVAD_LOW_FACTOR);
	float high_factor = verbio_get_curr_vad_param( speech, VVAD_HIGH_FACTOR);
	float final_factor = verbio_get_curr_vad_param( speech, VVAD_FINAL_FACTOR);
	float final_high_factor = verbio_get_curr_vad_param( speech, VVAD_FINAL_HIGH_FACTOR);
	float min_high_thresh = verbio_get_curr_vad_param( speech, VVAD_MIN_HIGH_THRESH);
	float aam_min = verbio_get_curr_vad_param( speech, VVAD_AAM_MIN);
	float aam_max = verbio_get_curr_vad_param( speech, VVAD_AAM_MAX);
	VAD_PRM prm;

	int dev = verbio_get_curr_dev( speech);
	
	if( dev == -2 || /* -1 reserved (to specify all channels)*/
			init_sil < 0 || 
			max_sil < 0 ||
			low_factor < 0 || 
			high_factor < 0 || 
			final_factor < 0 || 
			final_high_factor < 0 || 
			min_high_thresh < 0 || 
			aam_min < 0 ||
			aam_max < 0)
	{
		ast_log(LOG_ERROR, "Init params\n");
		return -1;
	}

	verbio_engine = speech->engine;

	/* ***** rec device ***** */
	vox_clrrsp( &rsp);
	rsp.maxsil = max_sil;
	rsp.initsil = init_sil;

	if( (recdev = vox_recstr_open( dev, &rsp, VERBIO_CODEC | MC_INITSIL )) == -1) 
	{
		ast_log(LOG_ERROR, "vox_recstr_open [%s]\n", ATVOX_ERRMSGP( dev));
		return -1;	
	}

	/* Store recdev in speech struct. We will need recdev later (when 
	   sending audio samples to recognizer)
	*/
	verbio_set_recdev( speech, recdev);

	/* ***** VAD ***** */
	vox_clrvad( &prm);
	prm.low_factor = low_factor;
	prm.high_factor = high_factor;
	prm.final_factor = final_factor;
	prm.final_high_factor = final_high_factor;
	prm.min_high_thresh = min_high_thresh;
	prm.aam_min = aam_min;
	prm.aam_max = aam_max;
	
	vsddev = vox_vsd_open( dev, recdev, &prm, VERBIO_CODEC);
	if( (int)vsddev == -1)
	{
		ast_log(LOG_ERROR, "vsddev\n");
		vox_recstr_close( dev, recdev);
		return -1;
	}

	/* Store vsddev in speech struct. We will need vsddev later (when
	   doing VAD.)
	*/
	verbio_set_vsddev( speech, vsddev);

	/* Init voice detection status*/
	verbio_set_voice_detected( speech, 0);

	/* set rec file (to record user input) */
	if( verbio_get_keep_rec_files( speech) > 0)
		if( verbio_set_rec_file( speech) < 0)
			ast_log(LOG_ERROR, "Error setting rec file.\n");

	/* Verbio engine is ready to accept samples */
	speech->state = AST_SPEECH_STATE_READY;

	return 0;
}

/*! \brief Change an engine specific attribute */
static int verbio_change(struct ast_speech *speech, char *name, const char *value)
{
	int verbose = verbio_get_verbose( speech);
	int retval = 0;

	if( verbose > 0 )
		ast_log(LOG_NOTICE, "Setting attribute \'%s\' to \'%s\'\n", 
									name, 
									value);
	if( !strcmp( name, "verbose"))
	{
		verbio_set_verbose( speech, atoi( value));
	}
	else if( !strcmp( name, "config"))
	{
		if( verbio_set_curr_config( speech, value) < 0)
			retval = -1;
	}
	else if( !strcmp( name, "language"))
	{
		if( verbio_set_curr_lang( speech, value) < 0)
			retval = -1;
	}
	else if( !strcmp( name, "lang"))
	{
		if( verbio_set_curr_lang( speech, value) < 0)
			retval = -1;
	}
	/*else if( !strcmp( name, "global_grammar"))
	{
		ast_log(LOG_NOTICE, "Setting global grammar %s.\n", value);
		if( verbio_set_global_grammar( speech, value) < 0)
			retval = -1;
	}*/
	else if( !strcmp( name, "init_sil"))
	{
		if( verbio_set_curr_vad_param( speech, VVAD_INIT_SIL, atof(value)) < 0)
			retval = -1;
	}
	else if( !strcmp( name, "max_sil"))
	{
		if( verbio_set_curr_vad_param( speech, VVAD_MAX_SIL, atof(value)) < 0)
			retval = -1;
	}
	else if( !strcmp( name, "min_ref"))
	{
		if( verbio_set_curr_vad_param( speech, VVAD_MAX_REF, atof(value)) < 0)
			retval = -1;
	}
	else if( !strcmp( name, "absolute_timeout"))
	{
		if( verbio_set_abs_timeout( speech, atoi(value)) < 0) 
			retval = -1;
	}
	else if( !strcmp( name, "low_factor"))
	{
		if( verbio_set_curr_vad_param( speech, VVAD_LOW_FACTOR, atof(value)) < 0)
			retval = -1;
	}
	else if( !strcmp( name, "high_factor"))
	{
		if( verbio_set_curr_vad_param( speech, VVAD_HIGH_FACTOR, atof(value)) < 0)
			retval = -1;	
	}
	else if( !strcmp( name, "final_factor"))
	{
		if( verbio_set_curr_vad_param( speech, VVAD_FINAL_FACTOR, atof(value)) < 0)
			retval = -1;	
	}
	else if( !strcmp( name, "final_high_factor"))
	{
		if( verbio_set_curr_vad_param( speech, VVAD_FINAL_HIGH_FACTOR, atof(value)) < 0)
			retval = -1;	
	}
	else if( !strcmp( name, "min_high_thresh"))
	{
		if( verbio_set_curr_vad_param( speech, VVAD_MIN_HIGH_THRESH, atof(value)) < 0)
			retval = -1;	
	}
	else if( !strcmp( name, "aam_min"))
	{
		if( verbio_set_curr_vad_param( speech, VVAD_AAM_MIN, atof(value)) < 0)
			retval = -1;	
	}
	else if( !strcmp( name, "aam_max"))
	{
		if( verbio_set_curr_vad_param( speech, VVAD_AAM_MAX, atof(value)) < 0)
			retval = -1;	
	}
	else if( !strcmp( name, "keep_recorded_files"))
	{
		verbio_set_keep_rec_files( speech, atoi(value));
	}
	else if( !strcmp( name, "other_atrribute"))
	{
		ast_log(LOG_NOTICE, "Setting other_atrributee to %s\n", value);
	}
	else
	{
		ast_log(LOG_WARNING, "Unknown attribute %s\n", name);
	}

	if( retval < 0)
		ast_log(LOG_ERROR, "Setting %s to %s\n", name, value);

	return retval;
}

/*! \brief  Change the type of results we want back  */
static int verbio_change_results_type(struct ast_speech *speech, enum ast_speech_results_type results_type)
{
	/*TODO: implementation */
	ast_log(LOG_NOTICE, "Verbio change results type not available yet\n");
	return 0;
}

/*! \brief Try to get results */
static struct ast_speech_result* verbio_get(struct ast_speech *speech)
{
	/*verbio_set_asr_result( speech);*/
	return speech->results;
}

struct ast_speech_engine verbio_engine = {
	.name = "verbio",
	.create = verbio_create,
	.destroy = verbio_destroy,
	.load = verbio_load,
	.unload = verbio_unload,
	.activate = verbio_activate,
	.deactivate = verbio_deactivate,
	.write = verbio_write,
	.dtmf = verbio_dtmf,
	.start = verbio_start,
	.change = verbio_change,
	.change_results_type = verbio_change_results_type,
	.get = verbio_get,
	.formats = AST_FORMAT_SLINEAR,
};

static int load_module(void)
{
	ast_log(LOG_NOTICE, "Loading Verbio Speech Technologies res speech\n");
	struct ast_config 	*vcfg;
	const char 		*vcfg_primary_host;
	const char 		*vcfg_default_asr_config;
	const char 		*vcfg_default_asr_lang;
	unsigned int		done = 0;
	const unsigned int	retries = 5;/*TODO: to config file*/

	vcfg = ast_config_load( VERBIO_CFG);

	if( !vcfg) 
	{
		ast_log(LOG_ERROR, "Error opening configuration file %s\n", 
								VERBIO_CFG);
		return -1;
	}

	/* If we perform a 'restart now' from CLI unload_module won't get called.
	   We need to disconnect prior to reconnect to verbio voxserver.
	*/
	ast_speech_unregister( verbio_engine.name);/*TODO: it's ok to do it?*/
	usleep( 300000);/*be careful with (fast) reconnections*/

	/* Set primary vox server ip*/
	if( !( vcfg_primary_host = ast_variable_retrieve(vcfg, "general", "primary_vox_server"))) 
		vcfg_primary_host = VERBIO_DEF_HOST;

	vox_setparm( -1, VXGB_DEFSERVER, vcfg_primary_host);

	if( !( vcfg_default_asr_config = ast_variable_retrieve(vcfg, "asr", "default_config"))) 
	{
		ast_log( LOG_ERROR, "ASR 'default_config' option missing\n");
		ast_config_destroy( vcfg);
		return 0;
	}

	if( !( vcfg_default_asr_lang = ast_variable_retrieve(vcfg, "asr", "default_language"))) 
	{
		ast_log( LOG_ERROR, "ASR 'default_language' option missing\n");
		ast_config_destroy( vcfg);
		return 0;
	}

	/* Init ASR */
	done = 0;
	unsigned int count = 0;
	while( (count < retries) && (done == 0))
	{
		if( vox_asr_init( vcfg_default_asr_config, vcfg_default_asr_lang) != -1) 
			done = 1;
		else	
			sleep( 1);
		++count;
	}	

	ast_config_destroy( vcfg);

	if( done == 1)
	{
		return ast_speech_register(&verbio_engine);
	}
	else 
	{
		ast_log(LOG_ERROR, "Could not connect to Verbio voxserver [%s]\n", 
							ATVOX_ERRMSGP( -1));
		return 0;/*do not stop asterisk startup*/
	}
}

static int unload_module(void)
{
	ast_log(LOG_NOTICE, "Unloading Verbio Speech Technologies res speech\n");
	ast_log(LOG_NOTICE, "Closing connection to Verbio voxserver..\n");
	vox_libclose();
	sleep( 1); /*be careful with (fast) reconnections*/

	return ast_speech_unregister(verbio_engine.name);
}

AST_MODULE_INFO_STANDARD(ASTERISK_GPL_KEY, "Verbio Speech Technologies ASR engine");
