/*
 * Asterisk -- An open source telephony toolkit.
 *
 * Copyright (C) <2007>, <Verbio Technologies>
 *
 * <Daniel Juan> <djuan@verbio.com>
 *
 * See http://www.asterisk.org for more information about
 * the Asterisk project. Please do not directly contact
 * any of the maintainers of this project for assistance;
 * the project provides a web site, mailing lists and IRC
 * channels for your use.
 *
 * This program is free software, distributed under the terms of
 * the GNU General Public License Version 2. See the LICENSE file
 * at the top of the source tree.
 */

/*! \file
 *
 * \brief Verbio Technologies Speech Applications
 *
 * \author Verbio Technologies <support@verbio.com> 
 *
 * \ingroup applications
 */

#include <stdio.h>
#include <stdlib.h>
#include <unistd.h>
#include <string.h>
#include <ctype.h> /*toupper*/

#include "asterisk.h"

ASTERISK_FILE_VERSION(__FILE__, "$Revision: 23899 $")

#include "asterisk/file.h"
#include "asterisk/logger.h"
#include "asterisk/channel.h"
#include "asterisk/pbx.h"
#include "asterisk/module.h"
#include "asterisk/lock.h"
#include "asterisk/app.h"
#include "asterisk/version.h"

/* Verbio voxlib */
#include <voxlib.h>

static const int 	VERBIO_BUFFER_SIZE 	= 1600;/* FIXME max 200ms audio */
static const int 	VERBIO_MAXINDEX 	= 32;
static const int 	VERBIO_MAX_RES_SIZE	= 100;
static const int 	VERBIO_MAX_CVAR_SIZE	= 120;
static const int 	VERBIO_MAX_INT_SIZE	= 10;
static const int 	VERBIO_MAX_TSTR_SIZE	= 10;
static const float 	VERBIO_MAX_REF		= 200.0;
static const float 	VERBIO_MIN_REF		= 5.0;
static const float	VERBIO_LOW_FACTOR	= 2.5;
static const float	VERBIO_HIGH_FACTOR	= 4.5;
static const float	VERBIO_FINAL_FACTOR	= 0.8;
static const float	VERBIO_FINAL_HIGH_FACTOR= 2.5;
static const float	VERBIO_MIN_HIGH_THRESH	= 500.0;
static const float	VERBIO_AAM_MIN		= 50.0;
static const float	VERBIO_AAM_MAX		= 200.0;

#define	VERBIO_CFG 		"verbio.conf"
#define	VERBIO_CODEC 		MC_ALAW
#define EXT 			"alaw"
#define VERBIO_DEF_HOST 	"127.0.0.1"
#define VERBIO_DEF_NET_TMOUT 	"5"
#define VERBIO_TTS_INIT_DELAY 	"650"
#define VERBIO_TTS_END_DELAY 	"80"
#define VERBIO_INIT_SIL 	"300"
#define VERBIO_MAX_SIL 		"200"
#define	VERBIO_DEF_ABS_TMOUT	"30"

#define AST_4 			10400

/* 
* Application info
*/
/* VerbioPrompt */
static char *verbio_prompt_descrip =
"VerbioPrompt(text_or_file[|lang][|speaker][|options])\n"
"Synthetise a text using Verbio TTS engine.\n"
"- text_or_file: text or file (see options) to synth\n"
"- lang        : tts language\n"
"- speaker     : tts speaker\n"
"- options     : v (verbosity on)\n"
"                f (tread text_or_file parameter as a text file to read from)\n"
"		 n (do not hangup on Verbio error)\n";
static char *verbio_prompt_app = "VerbioPrompt";

/* VerbioRec */
static char *verbio_rec_descrip =
"VerbioRec([|config][|lang][|initsil][|maxsil][|options])\n"
"Launch a recognition.\n"
"- config    : ASR config\n"
"- lang      : ASR language \n"
"- initsil   : Initial silence (max duration of initial silence 10ms units)\n"
"- maxsil    : Max silence (max duration of final silence 10ms units)\n"
"- options   : v (verbosity on)\n"
"              b (beep before recognition)\n"
"              d (enable dtmf detection)\n"
"	       n (do not hangup on Verbio error)\n\n"
"You /must/ execute VerbioLoadVcb app prior to Launch any recognition.\n\n"
"In order to check the recognition result, you need to check (once VerbioRec has finished) \n"
"the following channel vars:\n"
" - VASR_WORDS   : Number of recognized words (n).\n"
" - VASR_INDEXn  : Index (in grammar) of the n-word recognized.\n"
" - VASR_RESULTn : n-Result of recognition.\n"
" - VASR_SCOREn  : n-Score (confidence) of recognition.\n\n"
"For backward compatibility:\n"
" - VASR_INDEX  =  VASR_INDEX0\n"
" - VASR_RESULT =  VASR_RESULT0\n"
" - VASR_SCORE  =  VASR_SCORE0\n\n"
"If dtmf detection ('d' option) is enabled, the following channel vars will be set:\n"
" - VDTMF_DETECTED (TRUE or FALSE)\n"
" - VDTMF_RESULT (if VDTMF_DETECTED = TRUE, will contain the pressed key -0,1,2,3,4,5,6,7,8,9,*,#...-)\n";
static char *verbio_rec_app = "VerbioRec";

/* VerbioPromptAndRec */
static char *verbio_prompt_and_rec_descrip =
"VerbioPromptAndRec(text_or_file[|initsil][|maxsil][|tts_lang][|tts_spkr][|asr_conf][|asr_lang][|options])\n"
"VerbioPromptAndRec.\n"
"- text_or_file: text or file (see options) to synth\n"
"- initsil     : Initial silence (max duration of initial silence 10ms units)\n"
"- maxsil      : Max silence (max duration of final silence 10ms units)\n"
"- tts_lang    : TTS language\n"
"- tts_spkr    : TTS speaker\n"
"- asr_conf    : ASR config\n"
"- asr_lang    : ASR language \n"
"- options     : v (verbosity on)\n"
"                f (tread text_or_file parameter as a text file to read from)\n"
"                b (beep before rec -no sense when using bargein-)\n"
"                g (bargein activated -user can interrupt the system-)\n"
"                i (-if bargein is active- immediately stop tts on voice detection)\n"
"                d (enable dtmf detection)\n"
"	         n (do not hangup on Verbio error)\n\n"
"Please, when using bargein, be very carefull with initsil and maxsil parameters. If you do not\n"
"properly set them you may run into problems like recognition stopping\n"
"(prior to finish) our prompt (if user does not speak). \n"
"initsil must be high enough to allow the user to listen our (full) prompt.\n\n"
"Also remember that you /must/ execute VerbioLoadVcb app prior to launch a recognition.\n\n"
"In order to check the recognition result, you need to check (once VerbioPromptAndRec has finished) \n"
"the following channel vars:\n"
" - VASR_WORDS   : Number of recognized words (n).\n"
" - VASR_INDEXn  : Index (in grammar) of the n-word recognized.\n"
" - VASR_RESULTn : n-Result of recognition.\n"
" - VASR_SCOREn  : n-Score (confidence) of recognition.\n\n"
"For backward compatibility:\n"
" - VASR_INDEX  = VASR_INDEX0\n"
" - VASR_RESULT =  VASR_RESULT0\n"
" - VASR_SCORE  =  VASR_SCORE0\n\n"
"If dtmf detection ('d' option) is enabled, the following channel vars will be set:\n"
" - VDTMF_DETECTED (TRUE or FALSE)\n"
" - VDTMF_RESULT (if VDTMF_DETECTED = TRUE, will contain the pressed key -0,1,2,3,4,5,6,7,8,9,*,#...-)\n";
static char *verbio_prompt_and_rec_app = "VerbioPromptAndRec";

/* VerbioStreamAndRec */
static char *verbio_stream_and_rec_descrip =
"VerbioStreamAndRec(audio_file[|initsil][|maxsil][|asr_conf][|asr_lang][|options])\n"
"VerbioStreamAndRec.\n"
"- audio_file  : Audio file to stream.\n"
"- initsil     : Initial silence (max duration of initial silence 10ms units)\n"
"- maxsil      : Max silence (max duration of final silence 10ms units)\n"
"- asr_conf    : ASR config\n"
"- asr_lang    : ASR language \n"
"- options     : v (verbosity on)\n"
"                b (beep before rec -no sense when using bargein-)\n"
"                g (bargein activated -user can interrupt the system-)\n"
"                i (-if bargein is active- immediately stop stream on voice detection)\n"
"                d (enable dtmf detection)\n"
"	         n (do not hangup on Verbio error)\n\n"
"Please, when using bargein, be very carefull with initsil and maxsil parameters. If you do not\n"
"properly set them you may run into problems like recognition stopping\n"
"(prior to finish) our audio file (if user does not speak). \n"
"initsil must be high enough to allow the user to listen our (full) audio file.\n\n"
"Also remember that you /must/ execute VerbioLoadVcb app prior to launch a recognition.\n\n"
"In order to check the recognition result, you need to check (once VerbioPromptAndRec has finished) \n"
"the following channel vars:\n"
" - VASR_WORDS   : Number of recognized words (n).\n"
" - VASR_INDEXn  : Index (in grammar) of the n-word recognized.\n"
" - VASR_RESULTn : n-Result of recognition.\n"
" - VASR_SCOREn  : n-Score (confidence) of recognition.\n\n"
"For backward compatibility:\n"
" - VASR_INDEX  = VASR_INDEX0\n"
" - VASR_RESULT =  VASR_RESULT0\n"
" - VASR_SCORE  =  VASR_SCORE0\n\n"
"If dtmf detection ('d' option) is enabled, the following channel vars will be set:\n"
" - VDTMF_DETECTED (TRUE or FALSE)\n"
" - VDTMF_RESULT (if VDTMF_DETECTED = TRUE, will contain the pressed key -0,1,2,3,4,5,6,7,8,9,*,#...-)\n";
static char *verbio_stream_and_rec_app = "VerbioStreamAndRec";

/* VerbioLoadVcb */
static char *verbio_load_vcb_descrip =
"VerbioLoadVcb(gram_file|gram_type[|config][|lang][|options])\n"
"Load vocabulary.\n"
"- gram_file : grammar file\n"
"- gram_type : grammar type (ISOLATED or ABNF)\n"
"- config    : ASR config\n"
"- lang      : ASR language \n"
"- options   : v (verbosity on)\n"
"	       n (do not hangup on Verbio error)\n\n"
"This function will set a channel var (VVCB_HANDLE) containing the id (vcb_handle)\n"
"of the loaded vocabulary (see VerbioUnloadVcb to see why we need this handle).\n";

static char *verbio_load_vcb_app = "VerbioLoadVcb";

/* VerbioUnloadVcb */
static char *verbio_unload_vcb_descrip =
"VerbioUnloadVcb(vcb_handle[|config][|lang][|options])\n"
"Unload vocabulary.\n"
"- vcb_handle: vocabulary id to unload (-1 to unload all, and free licences)\n"
"- config    : ASR config \n"
"- lang      : ASR language \n"
"- options   : v (verbosity on)\n"
"	       n (do not hangup on Verbio error)\n\n"
"Please do not forget to execute this app on hangup...otherwise you may run out of licences.\n";
static char *verbio_unload_vcb_app = "VerbioUnloadVcb";

/* VerbioInfo */
static char *verbio_info_descrip =
"VerbioInfo()\n"
"Print Verbio-related info.\n";
static char *verbio_info_app = "VerbioInfo";

/* VerbioLastErr */
static char *verbio_last_err_descrip =
"VerbioLastErr(var)\n"
"Get Verbio last error.\n"
" - var: channel var where to store last error message.\n\n"
"You need to call Verbio apps using the 'n' option (in order to\n"
"disable hangup on app error).";
static char *verbio_last_err_app = "VerbioLastErr";


#if ASTERISK_VERSION_NUM < AST_4
static char *tdesc = "Verbio Speech Technologies Applications";

STANDARD_LOCAL_USER;
LOCAL_USER_DECL;
#endif

/* Helpler functions */
/*! \brief Helper function. Gets a valid id to be used as Verbio device 'dev'*/
static int verbio_get_dev( struct ast_channel *chan)
{
	int dev = 0;
	char tmpdev[10];
	
	/*This one, does not work when we have two concurrent 
	  -less than a second- calls
	  FIXME beware of 'systemname' option - asterisk.conf/asterisk 1.4 
	  (if set) We may get dev = 0 ...uniqid(1.4)='systemname-uniqid'
	  If we do not use chan->uniqueid.. this systemname option does not 
	  affect us (sure??)
	*/
	/*dev = atoi( chan->uniqueid);*/ 

	/* 
		Incremental index (number after dot (.) in 'uniqueid')
		FIXME: when this index gets reset? 
	*/
	strcpy( tmpdev, chan->uniqueid);
	char *dec = strchr( tmpdev,'.');
	dev = atoi( dec + 1);

	/* avoid negative devs..It's possible to have a negative value here?*/
	if( dev < 0)
		dev = -dev;

	return dev;
}

/*!\brief Helper function. Set last verbio error as a channel variable 'var'.*/
static void verbio_set_err( struct ast_channel *chan, int dev, const char *var)
{
	char *err_msg;
	long err_code;

	/*Get last verbio error*/
	err_code = ATVOX_LASTERR( dev);

	/*
	long   VERBIOAPI ATVOX_LASTERR(int dev);
	const char* VERBIOAPI ATVOX_ERRMSGP(int dev);
	
	 - :: Verbio error codes :: - 
	 (check voxlib.h header file for more info)

	 EVX_NOERROR        0	 NO ERROR 
	 EVX_INVSETUP     - 4	 Vox ERROR 
	 EVX_NOMEM        - 5	 OUT OF MEMORY.
	 EVX_VCBFILE      - 6	 THE VOCABULARY FILE NAME IS NOT VALID.
	 EVX_INVWORD      - 7	 THE VOCABULARY CONTAINS AN INVALID WORD.
	 EVX_NOLICFILE    - 8	 NO LICENSE FILE WAS FOUND. 
	 EVX_INVLIC       - 9	 THE LICENSE FILE IS NOT VALID. 
	 EVX_SYSTEM       -10	 SYSTEM ERROR (Check errno) 
	 EVX_NOLIBINIT    -13	 VOXLIB WAS NOT SUCCESSFULLY LOADED. 
	 EVX_NOLIC        -14	 NO LICENSE 
	 EVX_NOSETVCB     -15	 NO ACTIVE VOCABULARY. 
	 EVX_NORECSTR     -16	 NO RECOGNITION. 
	 EVX_NOLINE       -17	 NO LINES AVAILABLE FOR SPECIFIED CHAN DEVICE. 
	 EVX_BADPARM      -18	 INVALID PARAMETER IN FUNCTION CALL 
	 EVX_NOTIMP       -19	 NOT IMPLEMENTED 
	 EVX_NORECIND     -20	 NO RECIND OR NBEST. 
	 EVX_INVFILE      -21	 INVALID FILENAME 
	 EVX_NETWORK      -22	 NETWORK ERROR 
	 EVX_DICFILE      -23	 THE DICTIONARY FILE NAME IS NOT VALID 
	 EVX_PARSER       -24	 ABNF PARSER ERROR 
	 EVX_INVVER       -25	 CLIENT SERVER VERSION MISMATCH
	*/

	switch( err_code)
    	{
    	case 0:
	err_msg = "EVX_NOERROR";
        	break;
    	case -4:
	err_msg = "EVX_INVSETUP";
        	break;
    	case -5:
        err_msg = "EVX_NOMEM";
        	break;
    	case -6: 
       	err_msg = "EVX_VCBFILE";
        	break;
    	case -7: 
       	err_msg = "EVX_INVWORD";
        	break;
    	case -8: 
       	err_msg = "EVX_NOLICFILE";
        	break;
    	case -9: 
       	err_msg = "EVX_INVLIC";
        	break;
    	case -10: 
       	err_msg = "EVX_SYSTEM";
        	break;
    	case -13: 
       	err_msg = "EVX_NOLIBINIT";
        	break;
    	case -14: 
       	err_msg = "EVX_NOLIC";
        	break;
    	case -15: 
       	err_msg = "EVX_NOSETVCB";
        	break;
    	case -16: 
       	err_msg = "EVX_NORECSTR";
        	break;
    	case -17: 
       	err_msg = "EVX_NOLINE";
        	break;
    	case -18: 
       	err_msg = "EVX_BADPARM";
        	break;
    	case -19: 
       	err_msg = "EVX_NOTIMP";
        	break;
    	case -20: 
       	err_msg = "EVX_NORECIND";
        	break;
    	case -21: 
       	err_msg = "EVX_INVFILE";
        	break;
    	case -22: 
       	err_msg = "EVX_NETWORK";
        	break;
    	case -23: 
       	err_msg = "EVX_DICFILE";
        	break;
    	case -24: 
       	err_msg = "EVX_PARSER";
        	break;
    	case -25: 
       	err_msg = "EVX_INVVER";
        	break;
    	default: err_msg = "EVX_UNKNOWN";
    	}
	
	pbx_builtin_setvar_helper( chan, var, err_msg);
}

/* 
	TODO
	This function may be used in future versions, in order to 
	try to connect to a backup voxserver, when a call to voxlib fails...
	Nowadays primary and backup servers are only checked at module load.
*/
/*
static int verbio_check_connection( int dev, char *vcfg_primary_host, char *vcfg_backup_host)
{
	char *tmpparm;
	if( vox_getparm( dev, VXCH_SERVER, tmpparm) ==  -1)
	{
		vox_setparm( -1, VXCH_SERVER, vcfg_primary_host);
		if( vox_getparm( dev, VXCH_SERVER, tmpparm) ==  -1)
		{
			ast_log( LOG_WARNING, "Connection to primary voxserver is down %s.\n", vcfg_primary_host);
		
			vox_setparm( -1, VXCH_SERVER, vcfg_backup_host);
			if( vox_getparm( dev, VXCH_SERVER, tmpparm) ==  -1)
			{
				ast_log( LOG_ERROR, "Connection to backup voxserver is down %s.\n", vcfg_backup_host);
				return -1;
			}
		}
	}
	return 0;
}
*/

/*! \brief Helper function. Sets recognition result on Asterisk's dialplan (as channel var)*/
static int verbio_set_asr_result( struct ast_channel *chan, int dev)
{
	/* 
		This function will get the recognition (launched on 
		Verbio 'dev' channel) result from voxserver, and set
		it back to asterisk dialplan (as channel vars). 
	*/
	int 	nind = 0;
	int 	index[VERBIO_MAXINDEX+1];
	float 	score[VERBIO_MAXINDEX+1];
	char 	tmpbuff[VERBIO_MAX_RES_SIZE];
	char 	vtmpbuff[VERBIO_MAX_CVAR_SIZE];
	char 	itmpbuff[VERBIO_MAX_INT_SIZE];
	int 	size 	= 0;
	int 	i 	= 0;
	int 	scr 	= 0;
	
	/* -------------------------------------- */
	/* ----- Get result from recognizer ----- */
	/* -------------------------------------- */
	if(( nind = vox_recind( dev, VERBIO_MAXINDEX, index, score, 0)) < 0)
		ast_log( LOG_ERROR, "vox_recind %s\n", ATVOX_ERRMSGP( dev));

	/* ------------------------------------- */
	/* ----- Set Asterisk Channel vars ----- */
	/* ------------------------------------- */
	/*sprintf not safe (used snprintf)*/

	/* Number of recognized words */
	/*size = sprintf( tmpbuff, "%d",  nind);*/
	if( snprintf( tmpbuff, VERBIO_MAX_RES_SIZE, "%d",  nind) > VERBIO_MAX_RES_SIZE)
		ast_log( LOG_WARNING, "Number of recognized words may be truncated.\n");
	pbx_builtin_setvar_helper( chan, "VASR_WORDS", tmpbuff);

	/* Backward compatibility*/
	/*Index*/
	/*size = sprintf( tmpbuff, "%d",  index[0]);*/
	if( snprintf( tmpbuff, VERBIO_MAX_RES_SIZE, "%d",  index[0]) > VERBIO_MAX_RES_SIZE)
		ast_log( LOG_WARNING, "Index may be truncated.\n"); 
	pbx_builtin_setvar_helper( chan, "VASR_INDEX", tmpbuff);

	/*Result*/
	/*size = sprintf( tmpbuff, "%s", vox_word(dev, index[0]));*/
	if( snprintf( tmpbuff, VERBIO_MAX_RES_SIZE, "%s", vox_word(dev, index[0])) > VERBIO_MAX_RES_SIZE)
		ast_log( LOG_WARNING, "Result may be truncated.\n"); 
	pbx_builtin_setvar_helper( chan, "VASR_RESULT", tmpbuff);
	
	/*score*/
	scr = (int)score[0];
	/*size = sprintf( tmpbuff, "%d",  scr);*/
	if( snprintf( tmpbuff, VERBIO_MAX_RES_SIZE, "%d",  scr) > VERBIO_MAX_RES_SIZE)
		ast_log( LOG_WARNING, "Score may be truncated.\n"); 	
	pbx_builtin_setvar_helper( chan, "VASR_SCORE", tmpbuff);

	if( nind == 0) 
	{
		/* No words recognized */
		
		/*Index*/
		/*size = sprintf( tmpbuff, "%d",  index[0]);*/
		if( snprintf( tmpbuff, VERBIO_MAX_RES_SIZE, "%d",  index[0]) > VERBIO_MAX_RES_SIZE)
			ast_log( LOG_WARNING, "Index may be truncated.\n");
		pbx_builtin_setvar_helper( chan, "VASR_INDEX0", tmpbuff);
	
		/*Result*/
		/*size = sprintf( tmpbuff, "%s", vox_word(dev, index[0]));*/
		if( snprintf( tmpbuff, VERBIO_MAX_RES_SIZE, "%s", vox_word(dev, index[0])) > VERBIO_MAX_RES_SIZE)
			ast_log( LOG_WARNING, "Result may be truncated.\n"); 
		pbx_builtin_setvar_helper( chan, "VASR_RESULT0", tmpbuff);
		
		/*score*/
		size = sprintf( tmpbuff, "%d",  0);
		pbx_builtin_setvar_helper( chan, "VASR_SCORE0", tmpbuff);
		
	}
	else 
	{	/* for each recognized word...*/
		for( i = 0; i < nind; i++) 
		{
			/* convert our index to a char */
			/*size = sprintf( itmpbuff, "%d",  i);*/
			if( snprintf( itmpbuff, VERBIO_MAX_INT_SIZE, "%d",  i) > VERBIO_MAX_INT_SIZE)
				ast_log( LOG_WARNING, "Number of recognized words may be truncated.\n");

			/*create our var names (to be set in Ast. Dialplan)*/
			/* Index  */
			strcpy(vtmpbuff, "VASR_INDEX");
			strcat(vtmpbuff, itmpbuff);
			/* Set value */
			/*size = sprintf( tmpbuff, "%d",  index[i]);*/
			if( snprintf( tmpbuff, VERBIO_MAX_RES_SIZE, "%d",  index[i]) > VERBIO_MAX_RES_SIZE)
				ast_log( LOG_WARNING, "Index may be truncated.\n");
			pbx_builtin_setvar_helper( chan, vtmpbuff, tmpbuff);
			
			/*Result*/
			strcpy(vtmpbuff, "VASR_RESULT");
			strcat(vtmpbuff, itmpbuff);
			/*size = sprintf( tmpbuff, "%s", vox_word(dev, index[i]));*/
			if( snprintf( tmpbuff, VERBIO_MAX_RES_SIZE, "%s", vox_word(dev, index[i])) > VERBIO_MAX_RES_SIZE)
				ast_log( LOG_WARNING, "Result may be truncated.\n"); 
			pbx_builtin_setvar_helper( chan, vtmpbuff, tmpbuff);
			
			/*score*/
			strcpy(vtmpbuff, "VASR_SCORE");
			strcat(vtmpbuff, itmpbuff);
			scr = (int)score[i];
			/*size = sprintf( tmpbuff, "%d",  scr);*/
			if( snprintf( tmpbuff, VERBIO_MAX_RES_SIZE, "%d",  scr) > VERBIO_MAX_RES_SIZE)
				ast_log( LOG_WARNING, "Score may be truncated.\n");
			pbx_builtin_setvar_helper( chan, vtmpbuff, tmpbuff);
		}
	}
	return 0;
}

/*! \brief Helper function. Set dtmf detection result (as channel var)*/
static void verbio_set_dtmf_result( struct ast_channel *chan, struct ast_frame *f)
{
	char 	tmpbuff[VERBIO_MAX_RES_SIZE];
	/*int 	size 	= 0;*/

/*
	if( f->subclass == '#')
		ast_log(LOG_NOTICE, "# pressed.\n");
	else if( f->subclass == '*')
		ast_log(LOG_NOTICE, "* pressed.\n");
	else if( f->subclass == '1')
		ast_log(LOG_NOTICE, "1 pressed.\n");
	else if( f->subclass == '2')
		ast_log(LOG_NOTICE, "2 pressed.\n");
	else if( f->subclass == '3')
		ast_log(LOG_NOTICE, "3 pressed.\n");
	else if( f->subclass == '4')
		ast_log(LOG_NOTICE, "4 pressed.\n");
	else if( f->subclass == '5')
		ast_log(LOG_NOTICE, "5 pressed.\n");
	else if( f->subclass == '6')
		ast_log(LOG_NOTICE, "6 pressed.\n");
	else if( f->subclass == '7')
		ast_log(LOG_NOTICE, "7 pressed.\n");
	else if( f->subclass == '8')
		ast_log(LOG_NOTICE, "8 pressed.\n");
	else if( f->subclass == '9')
		ast_log(LOG_NOTICE, "9 pressed.\n");
	else if( f->subclass == '0')
		ast_log(LOG_NOTICE, "0 pressed.\n");
	else
		ast_log(LOG_NOTICE, "Unknown dtmf.\n");
*/

	ast_log(LOG_NOTICE, "Pressed %c.\n", f->subclass);

	/* Set result */
	pbx_builtin_setvar_helper( chan, "VDTMF_DETECTED", "TRUE");
	//size = sprintf( tmpbuff, "%c", f->subclass);
	if( snprintf( tmpbuff, VERBIO_MAX_RES_SIZE, "%c", f->subclass) > VERBIO_MAX_RES_SIZE) /*FIXME: only 1 char...tmpbuff too big*/
		ast_log( LOG_WARNING, "DTMF may be truncated.\n"); 
	pbx_builtin_setvar_helper( chan, "VDTMF_RESULT", tmpbuff);
}

/*! \brief Helper function. Reset dtmf detection channel var*/
static void verbio_reset_dtmf_result( struct ast_channel *chan)
{
	pbx_builtin_setvar_helper( chan, "VDTMF_DETECTED", "FALSE");
}

/* Verbio Apps */
/*! \brief Text to speech application. */
static int verbio_prompt( struct ast_channel *chan, void *data)
{
	#if ASTERISK_VERSION_NUM < AST_4
	struct 			localuser *u;
	#else
	struct 			ast_module_user *u;
	#endif
	/* Input options */
	char 			*options=NULL;
	char 			*text_or_file = NULL;
	char 			*args;
	int 			argc = 0;
	char			*argv[2];
	char			*opt;
	/*--*/
	int 			option_answer = 0;
	int 			option_no_hangup_on_verbio_err = 0;
	int 			option_asfile = 0;
	int 			option_verbose = 0;
	char 			*option_tts_lang;
	char 			*option_tts_spkr;
	/* Vars to store cfg file options*/
	const char 		*vcfg_primary_host;
	const char 		*vcfg_backup_host;
	const char 		*vcfg_default_tts_lang;
	const char 		*vcfg_default_tts_spkr;
	const char 		*vcfg_tts_txt_path;
	int			vcfg_keep_synthesized_files = 0;
	const char		*vcfg_recorded_files_path = NULL;
	/* Options to be used when dealing with voxlib*/
	const char		*tts_lang;
	const char		*tts_spkr;
	int 			dev = 0;
	int 			count 	= 0;

	/* = (char*)calloc( 1, VERBIO_BUFFER_SIZE);*/ /*FIXME:dynamic array??*/
	char			*samples[VERBIO_BUFFER_SIZE];
	
	char			*text_to_synth;
	/* Keep file */
	const int 		fflags 	= O_CREAT|O_TRUNC|O_WRONLY;
	struct 			ast_filestream *s 	= '\0';
	char 			filename[MAXPATHLEN] 	= "verbio-tts-";
	char 			rec_file_path[MAXPATHLEN];
	/* Helper vars*/
	char 			text_full_path[MAXPATHLEN];
	const char 		*tmp;
	struct ast_config 	*vcfg;
	struct ast_frame 	*f; /*audio data storage*/
	int 			vcfg_init_delay = 0;
	int 			vcfg_end_delay = 0;

	
	if( ast_strlen_zero( data)) 
	{
		ast_log( LOG_ERROR, "%s requires an argument (text or file"
					" |[lang]|[speaker]|[options])\n",
							verbio_prompt_app);
		return -1;
	}

	#if ASTERISK_VERSION_NUM < AST_4
	LOCAL_USER_ADD( u);
	#else
	u = ast_module_user_add(chan);
	#endif

	/* We need to make a copy of the input string if we are going to modify it! */
	args = ast_strdupa( data);	
	if( !args) 
	{
		ast_log( LOG_ERROR, "Out of memory!\n");
		#if ASTERISK_VERSION_NUM < AST_4
		LOCAL_USER_REMOVE( u);
		#else	
		ast_module_user_remove(u);
		#endif
		return -1;
	}
	
	if(( argc = ast_app_separate_args( args, '|', argv, sizeof(argv) / sizeof(argv[0])))) 
	{
		text_or_file = argv[0];
		options = argv[1];
	}

	/* --------------------------------- */
	/* ----- Get options from user ----- */
	/* --------------------------------- */
	/* <TTS lang> */
	option_tts_lang = strsep(&options, "|");
	if( option_tts_lang)
	{
		if( !strcmp( option_tts_lang, ""))
			option_tts_lang = NULL;
	}
	/* <TTS spkr> */
	option_tts_spkr = strsep(&options, "|");
	if( option_tts_spkr)
	{
		if( !strcmp( option_tts_spkr, ""))
			option_tts_spkr = NULL;
	}
	/* <Misc options> */
	opt = strsep(&options, "|");	
	if( opt) 
	{
		if( strchr( opt, 'f'))/*tread text_or_file as a file*/
			option_asfile = 1;
		if( strchr( opt, 'a'))
			option_answer = 1; 
		if( strchr( opt, 'n'))/*Do not hangup on verbio error*/
			option_no_hangup_on_verbio_err = 1; 
		if( strchr( opt, 'v'))
			option_verbose = 1;
	}

	/* ----------------------------------------------- */
	/* ----- Get options from Verbio config file ----- */
	/* ----------------------------------------------- */
	/* TODO: In future versions copy verbio settings to some kind of data 
		structure 
	*/
	vcfg = ast_config_load( VERBIO_CFG);
	/* Parse Verbio config*/
	if( !vcfg) 
	{
		ast_log( LOG_ERROR, "Error opening configuration file %s\n", VERBIO_CFG);
		#if ASTERISK_VERSION_NUM < AST_4
		LOCAL_USER_REMOVE( u);
		#else	
		ast_module_user_remove(u);
		#endif
		return -1;
	}
	if( !( vcfg_primary_host = ast_variable_retrieve( vcfg, "general", "primary_vox_server"))) 
		vcfg_primary_host = VERBIO_DEF_HOST;
	
	if( !( vcfg_backup_host = ast_variable_retrieve( vcfg, "general", "backup_vox_server"))) 	
		vcfg_backup_host = VERBIO_DEF_HOST;
	
	if( !( vcfg_default_tts_lang = ast_variable_retrieve( vcfg, "tts", "default_language"))) 
		ast_log( LOG_WARNING, "Error reading default_language option\n");
	
	if( !( vcfg_default_tts_spkr = ast_variable_retrieve( vcfg, "tts", "default_speaker"))) 
		ast_log( LOG_WARNING, "Error reading default_speaker option\n");
	
	if( !( vcfg_tts_txt_path = ast_variable_retrieve( vcfg, "tts", "text_prompts_path"))) 
		ast_log( LOG_WARNING, "Error reading text_prompts_path option\n");

	if( !( tmp = ast_variable_retrieve( vcfg, "tts", "init_delay"))) 
		tmp = VERBIO_TTS_INIT_DELAY;
	vcfg_init_delay = atoi( tmp);

	if( !( tmp = ast_variable_retrieve( vcfg, "tts", "end_delay"))) 
		tmp = VERBIO_TTS_END_DELAY;
	vcfg_end_delay = atoi( tmp);
	
	if( !( tmp = ast_variable_retrieve(vcfg, "debug", "keep_synthesized_files"))) 
		tmp = "0";
	vcfg_keep_synthesized_files = atoi( tmp);
	
	vcfg_recorded_files_path = ast_variable_retrieve(vcfg, "debug", "recorded_files_path");
	

	/* ---------------------------------------------------- */
	/* ----- Set options (to be used on voxlib calls) ----- */
	/* ---------------------------------------------------- */
	/* <language> */
	if( !option_tts_lang)
		tts_lang = vcfg_default_tts_lang;
	else
		tts_lang = option_tts_lang;
	/* <Speaker> */
	if( !option_tts_spkr)
		tts_spkr = vcfg_default_tts_spkr;
	else	
		tts_spkr = option_tts_spkr;
	/* <Text to read> */
	if( option_asfile)
	{
		/* <Text file> */
		if( text_or_file[0] != '/')/*FIXME: elegance...*/
		{	/* relative path */
			strcpy( text_full_path, vcfg_tts_txt_path);
			strcat( text_full_path, "/");	
			strcat( text_full_path, text_or_file);	
		}
		else
		{	/* full path */
			strcpy( text_full_path, text_or_file);
		}
		text_to_synth = ast_read_textfile( text_full_path);
	}
	else
	{
		text_to_synth = text_or_file;
	}
	/* <dev value> */
	dev = verbio_get_dev( chan);

	/* ---------------------------------------- */
	/* --- Save synthesized audio options  ---- */
	/* ---------------------------------------- */
	if( vcfg_keep_synthesized_files)
	{
		char t_str[VERBIO_MAX_TSTR_SIZE];
		time_t t = time( NULL );	
		//sprintf( t_str, "%d", (int)t);
		if( snprintf( t_str, VERBIO_MAX_TSTR_SIZE, "%d", (int)t) > VERBIO_MAX_TSTR_SIZE) 
			ast_log( LOG_WARNING, "Filename may be truncated.\n"); 
		strcat( filename, t_str);
		strcat( filename, "-");
		strcat( filename, chan->uniqueid);
		if( vcfg_recorded_files_path)
		{
			strcpy( rec_file_path, vcfg_recorded_files_path);
			strcat( rec_file_path, "/");
			strcat( rec_file_path, filename);
		}
		else
		{
			strcpy( rec_file_path, filename);
		}
	}
	
	/* <Print info to user...if (s)he wants> */
	if( option_verbose)
	{
		ast_log( LOG_NOTICE, "--------------------------\n");
		ast_log( LOG_NOTICE, "VerbioPrompt param summary:\n");
		ast_log( LOG_NOTICE, " Prim vox srv : %s\n", vcfg_primary_host);
		ast_log( LOG_NOTICE, " Bckp vox srv : %s\n", vcfg_backup_host);
		ast_log( LOG_NOTICE, " TTS language : %s\n", tts_lang);
		ast_log( LOG_NOTICE, " TTS speaker  :   %s\n", tts_spkr);
		if( option_asfile)
			ast_log( LOG_NOTICE, " Synth from %s text file\n", text_full_path);
		ast_log( LOG_NOTICE, " Text to synth: %s\n", text_to_synth);
		ast_log( LOG_NOTICE, " Init delay   : %d\n", vcfg_init_delay);
		ast_log( LOG_NOTICE, " End delay    : %d\n", vcfg_end_delay);
		if( vcfg_keep_synthesized_files)
			ast_log( LOG_NOTICE, " Rec file      :%s.%s\n", rec_file_path, EXT);
		
		ast_log( LOG_NOTICE, " Voxlib device: %d\n", dev);
		ast_log( LOG_NOTICE, "--------------------------\n");
	}
	
	/* ------------------ */
	/* Set channel format */
	/* ------------------ */
	/* Verbio only accepts alaw ulaw or slin...we are going to use alaw */
	if( ast_set_write_format( chan, AST_FORMAT_ALAW) <  0)
	{
		ast_log( LOG_ERROR, "AST_FORMAT_ALAW (write) failed.\n");
		#if ASTERISK_VERSION_NUM < AST_4
		LOCAL_USER_REMOVE( u);
		#else	
		ast_module_user_remove(u);
		#endif
		return -1;
	}
	if( ast_set_read_format( chan, AST_FORMAT_ALAW) <  0)
	{
		ast_log( LOG_ERROR, "AST_FORMAT_ALAW (read) failed.\n");
		#if ASTERISK_VERSION_NUM < AST_4
		LOCAL_USER_REMOVE( u);
		#else	
		ast_module_user_remove(u);
		#endif
		return -1;

	}

	/* ---------------------------- */
	/* ----- Verbio TTS stuff ----- */
	/* ---------------------------- */
	/* If we can not set tts lang.. exit*/
	if( vox_setparm( dev,VXCH_DEFTTSLANG, tts_lang))
	{
		ast_log( LOG_ERROR, "Can not set TTS language to %s.\n", tts_lang);
		#if ASTERISK_VERSION_NUM < AST_4
		LOCAL_USER_REMOVE( u);
		#else	
		ast_module_user_remove(u);
		#endif
		if( option_no_hangup_on_verbio_err)
			return 0;
		else
			return -1;
	}
	/* Speaker is not critical */
	if( vox_setparm( dev,VXCH_TTSSPKNAME, tts_spkr))
		ast_log( LOG_WARNING, "Can not set TTS speaker to %s.\n", tts_spkr);
	

	/*Hardcoded..if we want to change any of these tts params...SSML tags*/
	/*Set volume*/
 	int volume = 20000;
	vox_setparm(dev, VXCH_TTSVOLUME, &volume);

	/*Set speed*/
	int speed = 0; /* default*/
	vox_setparm(dev, VXCH_TTSSPEED, &speed);

	/*Set pitch*/
	int pitch = 0; /* default*/
	vox_setparm(dev, VXCH_TTSPITCH, &pitch);

	int synth = vox_playstr_open( dev, text_to_synth, VERBIO_CODEC);/*TODO:allow other formats? (ulaw and lin16)*/
	if( !synth)
	{
		ast_log( LOG_ERROR, "Fatal error on playstr_open %s\n", ATVOX_ERRMSGP( dev));
		#if ASTERISK_VERSION_NUM < AST_4
		LOCAL_USER_REMOVE( u);
		#else	
		ast_module_user_remove(u);
		#endif
		if( option_no_hangup_on_verbio_err)
			return 0;
		else
			return -1;
	}

	/* Answer the channel (if not up)*/
	int res = 0;
	if( chan->_state != AST_STATE_UP) 
	{ 
		if( option_answer) 
		{
			res = ast_answer(chan);
		} 
		else 
		{
			/* At the user's option, skip if the line is not up */
			#if ASTERISK_VERSION_NUM < AST_4
			LOCAL_USER_REMOVE( u);
			#else	
			ast_module_user_remove(u);
			#endif
			return -1;
		}
	}
	
	if( res) 
	{
		ast_log(LOG_WARNING, "Could not answer channel '%s'\n", chan->name);
		#if ASTERISK_VERSION_NUM < AST_4
		LOCAL_USER_REMOVE( u);
		#else	
		ast_module_user_remove(u);
		#endif
		return -1;
	}

	/* keep a copy of synth audio */
	if( vcfg_keep_synthesized_files)
		s = ast_writefile( rec_file_path, EXT, NULL, fflags , 0, 0644);
	

	/* Ensure no streams are currently running..FIXME:necessary? */
	ast_stopstream( chan);

	/*while( ast_waitfor( chan, -1) < -1)
		f = ast_read( chan);*/
	/* FIXME: First samples corrupted? */
	/* Initial delay */
	ast_safe_sleep( chan, vcfg_init_delay);

	while( 1)
	{
		if( ast_waitfor( chan, -1) < 0)
		{
			ast_log(LOG_NOTICE, "Wait failed.\n");
			vox_playstr_close( dev, synth);
			ast_stopstream(chan);
			#if ASTERISK_VERSION_NUM < AST_4
			LOCAL_USER_REMOVE( u);
			#else	
			ast_module_user_remove(u);
			#endif
			return -1;	
		}

		/* Read a frame... we will reuse this frame for writing.
		   FIXME:create a new frame...?*/
		f = ast_read( chan);
		/* Hangup detection */
		if( !f)
		{
			ast_log(LOG_NOTICE, "Hangup detected.\n");
			vox_playstr_close( dev, synth);
			ast_stopstream(chan);
			
			#if ASTERISK_VERSION_NUM < AST_4
			LOCAL_USER_REMOVE( u);
			#else	
			ast_module_user_remove(u);
			#endif
			return -1;	
		}
		
		if( f->frametype == AST_FRAME_VOICE) 
		{
			/* Get a number (f->samples) of synth audio */
			/*TODO: avoid blocking functions */
			count = vox_playstr_read( synth, samples, f->samples);
		
			/* Tell the frame which are it's new samples */
			f->data = samples;

			 /* user also wants to write to a file*/
			if( vcfg_keep_synthesized_files)
			{
				if( ast_writestream( s, f)) 
				{
					ast_log(LOG_WARNING, "Problem writing frame\n");
					ast_frfree(f);
					/*break;*/
				}
			}
			/* Write frame to chan */
			/*ast_queue_frame( chan, f);*//* internally frees f*/
			if( ast_write(chan, f))
			{ 
				/*TODO: handle this case*/
				ast_frfree(f);
				/*break;*/
			}
		} 
		ast_frfree( f);
		
		/* Check if we are done */
		if( count < f->samples)
			break;
		
	}
	 	
	/* FIXME: Last samples (sometimes) corrupted? */
	/* End delay */ 
	ast_safe_sleep( chan, vcfg_end_delay);

	ast_config_destroy( vcfg);

	/*Voxlib cleanup*/
	vox_playstr_close( dev, synth);
	
	ast_stopstream(chan);

	#if ASTERISK_VERSION_NUM < AST_4
	LOCAL_USER_REMOVE( u);
	#else	
	ast_module_user_remove(u);
	#endif
	return 0;
}

/*! \brief Recognition application. */
static int verbio_rec( struct ast_channel *chan, void *data)
{
	int 	res 	= 0;
	#if ASTERISK_VERSION_NUM < AST_4
	struct 			localuser *u;
	#else
	struct 			ast_module_user *u;
	#endif
	char 	*options=NULL;
	char 	*args 	= NULL;
	int 	argc 	= 0;
	char 	*argv[1];
	struct ast_config *vcfg;
	char	*opt 	= NULL;
	/* Config file opts */
	const char 	*vcfg_default_asr_config= NULL;
	const char 	*vcfg_default_asr_lang 	= NULL;
	const char 	*vcfg_primary_host	= NULL;
	const char 	*vcfg_backup_host	= NULL;
	int 		vcfg_keep_recorded_files= 0;
	const char	*vcfg_recorded_files_path = NULL;
	const char 	*vcfg_max_sil 		= NULL;
	const char 	*vcfg_init_sil		= NULL;
	const char	*vcfg_max_ref		= NULL;	
	const char 	*vcfg_min_ref		= NULL;
	int 		vcfg_absolute_timeout 	= 0;
	/* Input Options */
	int 	option_no_hangup_on_verbio_err = 0;
	int	option_answer		= 0;
	char	*option_init_sil	= NULL;
	char 	*option_max_sil 	= NULL;
	int 	option_beep 		= 0;
	int 	option_dtmf		= 0;
	int 	option_verbose 		= 0;
	char	*option_asr_config 	= NULL;
	char	*option_asr_lang 	= NULL;
	/* Vars to be used when dealing with voxlib */
	int 	dev 	= 0;
	const char	*asr_config 	= NULL;
	const char	*asr_lang 	= NULL;
	int 	n 	= 0;
	int 	recdev 	= 0;
	VX_RSP 	rsp;
	int 	init_sil	= 300;
	int 	max_sil		= 200;
	struct ast_frame *f;
	float	max_ref		= 0.0;
	float 	min_ref		= 0.0;
	/* Keep recorded files related vars */
	int 	fflags 			= O_CREAT|O_TRUNC|O_WRONLY;
	struct 	ast_filestream *s 	= '\0';
	char 	filename[MAXPATHLEN] 	= "verbio-rec-";
	char 	rec_file_path[MAXPATHLEN];
	/* Misc vars*/
	const char 	*tmp;
	

	#if ASTERISK_VERSION_NUM < AST_4
	LOCAL_USER_ADD( u);
	#else
	u = ast_module_user_add(chan);
	#endif
	
	/* We need to make a copy of the input string if we are going to modify it! */
	args = ast_strdupa(data);	
	if( !args) 
	{
		ast_log(LOG_ERROR, "Out of memory!\n");
		#if ASTERISK_VERSION_NUM < AST_4
		LOCAL_USER_REMOVE( u);
		#else	
		ast_module_user_remove(u);
		#endif
		return -1;
	}
	
	if(( argc = ast_app_separate_args(args, '|', argv, sizeof(argv) / sizeof(argv[0]))))
		options = argv[0];


	/* --------------------------------- */
	/* ----- Get options from user ----- */
	/* --------------------------------- */
	/* <ASR config> */
	option_asr_config = strsep(&options, "|");
	if( option_asr_config)
	{
		if( !strcmp( option_asr_config, ""))
			option_asr_config = NULL;
	}
	/* <ASR lang> */
	option_asr_lang = strsep( &options, "|");
	if( option_asr_lang)
	{
		if( !strcmp( option_asr_lang, ""))
			option_asr_lang = NULL;
	}
	/* <init sil> */
	option_init_sil = strsep(&options, "|");
	if( option_init_sil)
	{
		if( !strcmp( option_init_sil, ""))
			option_init_sil = NULL;
	}
	/* <max sil> */
	option_max_sil = strsep(&options, "|");
	if( option_max_sil)
	{
		if( !strcmp( option_max_sil, ""))
			option_max_sil = NULL;
	}
	/* <Misc options> */
	opt = strsep( &options, "|");	
	if( opt) 
	{
		if( strchr( opt, 'v'))
			option_verbose = 1;
		if( strchr( opt, 'a'))
			option_answer = 1;
		if( strchr( opt, 'n'))
			option_no_hangup_on_verbio_err = 1;
		if( strchr( opt, 'b'))
			option_beep = 1;/*play a beep before rec*/
		if( strchr( opt, 'd'))
			option_dtmf = 1; /* enable dtmf detection*/	
		
	}
	
	/* ----------------------------------------------- */
	/* ----- Get options from Verbio config file ----- */
	/* ----------------------------------------------- */
	vcfg = ast_config_load( VERBIO_CFG);
	if( !vcfg) 
	{
		ast_log(LOG_WARNING, "Error opening configuration file %s\n", VERBIO_CFG);
		#if ASTERISK_VERSION_NUM < AST_4
		LOCAL_USER_REMOVE( u);
		#else	
		ast_module_user_remove(u);
		#endif
		return -1;
	}

	if( !( vcfg_primary_host = ast_variable_retrieve( vcfg, "general", "primary_vox_server"))) 
		vcfg_primary_host = VERBIO_DEF_HOST;

	if( !( vcfg_backup_host = ast_variable_retrieve( vcfg, "general", "backup_vox_server"))) 	
		vcfg_backup_host = VERBIO_DEF_HOST;

	if( !( vcfg_default_asr_config = ast_variable_retrieve(vcfg, "asr", "default_config"))) 
		ast_log( LOG_WARNING, "Error reading default_config option\n");

	if( !( vcfg_default_asr_lang = ast_variable_retrieve(vcfg, "asr", "default_language"))) 
		ast_log( LOG_WARNING, "Error reading default_language option\n");
	
	if( !( vcfg_init_sil = ast_variable_retrieve(vcfg, "asr", "init_sil"))) 
		vcfg_init_sil = VERBIO_INIT_SIL;

	if( !( vcfg_max_sil = ast_variable_retrieve(vcfg, "asr", "max_sil"))) 
		vcfg_max_sil = VERBIO_MAX_SIL;
	
	if( !( vcfg_max_ref = ast_variable_retrieve(vcfg, "asr", "max_ref"))) 
		vcfg_max_ref = NULL;
	
	if( !( vcfg_min_ref = ast_variable_retrieve(vcfg, "asr", "min_ref"))) 
		vcfg_min_ref = NULL;

	if( !( tmp = ast_variable_retrieve(vcfg, "asr", "absolute_timeout"))) 
		tmp = VERBIO_DEF_ABS_TMOUT;
	vcfg_absolute_timeout = atoi( tmp);

	if( !( tmp = ast_variable_retrieve(vcfg, "debug", "keep_recorded_files"))) 
		tmp = "0";
	vcfg_keep_recorded_files = atoi( tmp);
	
	vcfg_recorded_files_path = ast_variable_retrieve(vcfg, "debug", "recorded_files_path");
	

	/* ---------------------------------------------------- */
	/* ----- Set options (to be used on voxlib calls) ----- */
	/* ---------------------------------------------------- */
	/* <ASR config> */
	if( !option_asr_config)
		asr_config = vcfg_default_asr_config;
	else
		asr_config = option_asr_config;
	/* <ASR lang> */
	if( !option_asr_lang)
		asr_lang = vcfg_default_asr_lang;
	else	
		asr_lang = option_asr_lang;
	/* <init sil> */
	if( !option_init_sil)
		init_sil = atoi( vcfg_init_sil);
	else
		init_sil = atoi( option_init_sil);
	/* <max sil> */
	if( !option_max_sil)
		max_sil = atoi( vcfg_max_sil);
	else	
		max_sil = atoi( option_max_sil);
	/* <dev value> */
	dev = verbio_get_dev( chan);

	
	/* ---------------------------------------- */
	/* ----- Save recorded audio options  ----- */
	/* ---------------------------------------- */
	if( vcfg_keep_recorded_files)
	{
		char t_str[VERBIO_MAX_TSTR_SIZE];
		time_t t = time( NULL );	
		//sprintf( t_str, "%d", (int)t);
		if( snprintf( t_str, VERBIO_MAX_TSTR_SIZE, "%d", (int)t) > VERBIO_MAX_TSTR_SIZE) 
			ast_log( LOG_WARNING, "Filename may be truncated.\n"); 
		strcat(filename, t_str);
		strcat(filename, "-");
		strcat(filename, chan->uniqueid);
		if( vcfg_recorded_files_path)
		{
			strcpy( rec_file_path, vcfg_recorded_files_path);
			strcat( rec_file_path, "/");
			strcat( rec_file_path, filename);
		}
		else
		{
			strcpy( rec_file_path, filename);
		}
	}
	/* <Print info to user...if (s)he wants> */
	if( option_verbose)
	{
		ast_log( LOG_NOTICE, "------------------------\n");
		ast_log( LOG_NOTICE, "VerbioRec param summary:\n");
		ast_log( LOG_NOTICE, " Prim vox srv : %s\n", vcfg_primary_host);
		ast_log( LOG_NOTICE, " Bckp vox srv : %s\n", vcfg_backup_host);
		ast_log( LOG_NOTICE, " ASR config   : %s\n", asr_config);
		ast_log( LOG_NOTICE, " ASR lang     :   %s\n", asr_lang);
		ast_log( LOG_NOTICE, " Init sil     :   %d\n", init_sil);
		ast_log( LOG_NOTICE, " Max  sil     :   %d\n", max_sil);
		ast_log( LOG_NOTICE, " Abs timeout  : %d\n", vcfg_absolute_timeout);
		if( vcfg_max_ref)
			ast_log( LOG_NOTICE, " Max  ref     : %f\n", atof( vcfg_max_ref));
		if( vcfg_min_ref)
			ast_log( LOG_NOTICE, " Min  ref     : %f\n", atof( vcfg_min_ref));
		ast_log( LOG_NOTICE, " Voxlib device: %d\n", dev);
		if( vcfg_keep_recorded_files)
			ast_log( LOG_NOTICE, " Rec file      :%s.%s\n", rec_file_path, EXT);
		ast_log( LOG_NOTICE, "------------------------\n");
	}

	/* ----------------------------- */
	/* Answer the channel (if not up)*/
	/* ----------------------------- */
	if( chan->_state != AST_STATE_UP) 
	{
		if( option_answer) 
		{
			res = ast_answer(chan);
			
		} 
		else 
		{
			/* At the user's option, skip if the line is not up */
			#if ASTERISK_VERSION_NUM < AST_4
			LOCAL_USER_REMOVE( u);
			#else	
			ast_module_user_remove(u);
			#endif
			return -1;
		}
			
	}
	
	if( res) 
	{
		ast_log(LOG_WARNING, "Could not answer channel '%s'\n", chan->name);
		#if ASTERISK_VERSION_NUM < AST_4
		LOCAL_USER_REMOVE( u);
		#else	
		ast_module_user_remove(u);
		#endif
		return -1;
	}

	/* ------------------ */
	/* Set channel format */
	/* ------------------ */
	/* Verbio only accepts alaw ulaw and slin...we are going to use alaw*/
	if( ast_set_write_format(chan, AST_FORMAT_ALAW) <  0)
	{
		ast_log(LOG_NOTICE, "AST_FORMAT_ALAW (write) failed.\n");
		#if ASTERISK_VERSION_NUM < AST_4
		LOCAL_USER_REMOVE( u);
		#else	
		ast_module_user_remove(u);
		#endif
		return -1;
	}
	if( ast_set_read_format(chan, AST_FORMAT_ALAW) <  0)
	{
		ast_log(LOG_NOTICE, "AST_FORMAT_ALAW (read) failed.\n");
		#if ASTERISK_VERSION_NUM < AST_4
		LOCAL_USER_REMOVE( u);
		#else	
		ast_module_user_remove(u);
		#endif
		return -1;
	}

	/* ---------------------------- */
	/* ----- Verbio Rec stuff ----- */
	/* ---------------------------- */
	vox_clrrsp( &rsp);
	rsp.maxsil = max_sil;
	rsp.initsil = init_sil;

	/* If we can not set asr config.. exit*/
	if( vox_setparm( dev,VXCH_DEFASRCFG, asr_config))
	{
		ast_log( LOG_ERROR, "Can not set ASR config to %s. [%s]\n", asr_config, ATVOX_ERRMSGP( dev));
		#if ASTERISK_VERSION_NUM < AST_4
		LOCAL_USER_REMOVE( u);
		#else	
		ast_module_user_remove(u);
		#endif
		if( option_no_hangup_on_verbio_err)
			return 0;
		else
			return -1;	
	}
	/* If we can not set asr lang.. exit*/
	if( vox_setparm( dev,VXCH_DEFASRLNG, asr_lang))
	{
		ast_log( LOG_ERROR, "Can not set ASR lang to %s. [%s]\n", asr_lang, ATVOX_ERRMSGP( dev));
		#if ASTERISK_VERSION_NUM < AST_4
		LOCAL_USER_REMOVE( u);
		#else	
		ast_module_user_remove(u);
		#endif
		if( option_no_hangup_on_verbio_err)
			return 0;
		else
			return -1;	
	}
	/* Open recognition stream */
	if( (recdev = vox_recstr_open( dev, &rsp, VERBIO_CODEC | MC_INITSIL )) == -1) 
	{
		ast_log(LOG_ERROR, "vox_recstr_open. [%s]\n", ATVOX_ERRMSGP( dev));
		#if ASTERISK_VERSION_NUM < AST_4
		LOCAL_USER_REMOVE( u);
		#else	
		ast_module_user_remove(u);
		#endif
		if( option_no_hangup_on_verbio_err)
			return 0;
		else
			return -1;	
	}

	/* maxref  VAD */
	if( vcfg_max_ref != NULL)
	{
		max_ref = atoi( vcfg_max_ref);
		if( max_ref > VERBIO_MAX_REF)
			max_ref = VERBIO_MAX_REF;
			
		if( vox_setparm( dev, VXGB_VSDMAXREF, &max_ref) == -1) 
			ast_log( LOG_ERROR, "Error setting max_ref.\n");
	}

	/* minref  VAD */
	if( vcfg_min_ref != NULL)
	{
		min_ref = atoi( vcfg_min_ref);
		if( min_ref > VERBIO_MIN_REF)
			min_ref = VERBIO_MIN_REF;

		if( vox_setparm( dev, VXGB_VSDMINREF, &min_ref) == -1)
			ast_log( LOG_ERROR, "Error setting min_ref.\n");	
	}

	/* keep a copy of what speaker says and let the user know (via a chan var)
	   which file corresponds to actual recording */
	if( vcfg_keep_recorded_files)
	{
		s = ast_writefile( rec_file_path, EXT, NULL, fflags , 0, 0644);
		pbx_builtin_setvar_helper( chan, "VASR_REC_FILE", rec_file_path);
	}
	
	if( option_beep) 
	{
		/* Some code to play a nice little beep to signify the start of the record operation */
		res = ast_streamfile( chan, "beep", chan->language);
		if( !res) 
			res = ast_waitstream( chan, "");
		else 
			ast_log(LOG_WARNING, "ast_streamfile failed on %s\n", chan->name);
		ast_stopstream(chan);
	}

	/* Ensure no streams are currently running.. FIXME: necessary? */
	ast_stopstream( chan);

	/* Recognition loop */
	verbio_reset_dtmf_result( chan);/* Avoid confusion among recognitions*/
	int sample_count = 0;
	while( 1)
	{
		if( ast_waitfor( chan, -1) < 0)
		{
			ast_log(LOG_NOTICE, "Wait failed.\n");
			vox_recstr_close( dev, recdev);
			ast_stopstream(chan);
			#if ASTERISK_VERSION_NUM < AST_4
			LOCAL_USER_REMOVE( u);
			#else	
			ast_module_user_remove(u);
			#endif
			return -1;	
		}

		/* Read a frame*/
		f = ast_read( chan);
		if( !f)
		{
			ast_log( LOG_NOTICE, "Hangup detected.\n");
 			vox_recstr_close( dev, recdev);
			ast_stopstream(chan);
			#if ASTERISK_VERSION_NUM < AST_4
			LOCAL_USER_REMOVE( u);
			#else	
			ast_module_user_remove(u);
			#endif
 			return -1;
		}
		
		if( f->frametype == AST_FRAME_VOICE)
		{
			/* Write frame to file */
			if( vcfg_keep_recorded_files)
			{
				if( ast_writestream( s, f)) 
				{
					ast_log(LOG_WARNING, "Problem writing frame\n");
					ast_frfree(f);
					/*break;*/
				}
			}
			/* Pass our frame to recognizer */
			/*TODO: avoid blocking functions */
			n = vox_recstr_write( recdev, f->data, f->samples);

			/* Check if we are done */
			if( n < f->samples)
				break;
			
			/* Absolute timeout counter */
			sample_count += f->samples;
			if( sample_count > (vcfg_absolute_timeout * 8000))
			{
				ast_log(LOG_WARNING, "Absolute timeout reached.\n");
				break; 
			}
		}
		else if( f->frametype == AST_FRAME_DTMF) 
		{
			if( option_dtmf)
			{
				/* We are done ... set detected dtmf to dialplan var*/
				verbio_set_dtmf_result( chan, f);
				ast_log(LOG_NOTICE, "DTMF detected.\n");
				break; 
			}
		}
	}

	/* Get and set (as chan vars) asr result */
	verbio_set_asr_result( chan, dev);

	ast_config_destroy( vcfg);

	/* Cleanup */
	vox_recstr_close( dev, recdev);
	
	ast_stopstream(chan);
	
	#if ASTERISK_VERSION_NUM < AST_4
	LOCAL_USER_REMOVE( u);
	#else	
	ast_module_user_remove(u);
	#endif
	return res;

}

/*! \brief Prompt and recognition function (implements bargein). */
static int verbio_prompt_and_rec( struct ast_channel *chan, void *data)
{
	/*
		This function allows us to launch a promt and a recognition at 
		the same time. As soon as the recognizer has some result for us,
		prompt will be stoped (if bargein option is active).
	*/
	#if ASTERISK_VERSION_NUM < AST_4
	struct 			localuser *u;
	#else
	struct 			ast_module_user *u;
	#endif
	/* Input options */
	char 	*options	= NULL;
	char 	*text_or_file 	= NULL;
	char 	*args;
	int 	argc = 0;
	char	*argv[2];
	char	*opt;
	/*--*/
	int 	option_answer 		= 0;
	int	option_no_hangup_on_verbio_err = 0;
	int 	option_asfile 		= 0;
	int 	option_verbose 		= 0;
	int	option_bargein 		= 0;
	int 	option_interrupt	= 0;
	int 	option_beep 		= 0;
	int 	option_dtmf		= 0;
	char 	*option_tts_lang	= NULL;
	char 	*option_tts_spkr	= NULL;
	char	*option_init_sil	= NULL;
	char 	*option_max_sil 	= NULL;
	char	*option_asr_config 	= NULL;
	char	*option_asr_lang 	= NULL;
	/* Vars to store cfg file options*/
	const char 	*vcfg_primary_host	= NULL;
	const char 	*vcfg_backup_host	= NULL;
	const char 	*vcfg_default_tts_lang	= NULL;
	const char 	*vcfg_default_tts_spkr	= NULL;
	const char 	*vcfg_tts_txt_path	= NULL;
	int 		vcfg_keep_recorded_files= 0;
	const char 	*vcfg_default_asr_config= NULL;
	const char 	*vcfg_default_asr_lang 	= NULL;
	const char 	*vcfg_max_sil 		= NULL;
	const char 	*vcfg_init_sil		= NULL;
	int		vcfg_keep_synthesized_files = 0;
	const char	*vcfg_recorded_files_path = NULL;
	const char	*vcfg_max_ref		= NULL;	
	const char 	*vcfg_min_ref		= NULL;
	int 		vcfg_absolute_timeout 	= 0;
	/* VAD Client-side params*/
	const char	*vcfg_low_factor	= NULL;
	const char	*vcfg_high_factor	= NULL;
	const char	*vcfg_final_factor	= NULL;
	const char	*vcfg_final_high_factor	= NULL;
	const char	*vcfg_min_high_thresh	= NULL;
	const char	*vcfg_aam_min		= NULL;
	const char	*vcfg_aam_max		= NULL;
	/* Options to be used when dealing with voxlib*/
	const char	*tts_lang	= NULL;
	const char	*tts_spkr	= NULL;
	int 		dev 		= 0;
	int 		count 		= 0;
	char		*samples[VERBIO_BUFFER_SIZE];
	char		*text_to_synth	=  NULL;
	const char	*asr_config 	= NULL;
	const char	*asr_lang 	= NULL;
	int 		recdev 		= 0;
	int 		vsddev 		= 0;
	VX_RSP 	rsp;
	int 	init_sil= 300;
	int 	max_sil	= 200;
	int	max_ref	= 0;
	int 	min_ref	= 0;
	float 	low_factor = 0.0;
	float 	high_factor = 0.0;
	float 	final_factor = 0.0;
	float 	final_high_factor = 0.0;
	float 	min_high_thresh = 0.0;
	float 	aam_min = 0.0;
	float 	aam_max = 0.0;

	/* Keep files */
	const int 	fflags 			= O_CREAT|O_TRUNC|O_WRONLY;
	struct 		ast_filestream *s_tts 	= '\0';
	struct 		ast_filestream *s_asr 	= '\0';
	char 		filename_tts[MAXPATHLEN] 	= "verbio-tts-";
	char 		filename_asr[MAXPATHLEN] 	= "verbio-rec-";
	char 		rec_tts_file_path[MAXPATHLEN];
	char 		rec_asr_file_path[MAXPATHLEN];
	/* Helper vars*/
	char 		text_full_path[MAXPATHLEN];
	const char 	*tmp;
	struct 		ast_config 	*vcfg;
	struct 		ast_frame *f; /*audio data storage*/
	int 		vcfg_init_delay = 0;
	int 		vcfg_end_delay = 0;
	
	
	if( ast_strlen_zero(data)) 
	{
		ast_log( LOG_WARNING, "%s requires an argument (text_or_file[|initsil][|maxsil][|tts_lang][|tts_spkr][|asr_conf][|asr_lang][|options])\n",verbio_prompt_and_rec_app);
		return -1;
	}
	
	#if ASTERISK_VERSION_NUM < AST_4
	LOCAL_USER_ADD( u);
	#else
	u = ast_module_user_add(chan);
	#endif
	
	/* We need to make a copy of the input string if we are going to modify it! */
	args = ast_strdupa( data);	
	if( !args) 
	{
		ast_log( LOG_ERROR, "Out of memory!\n");
		#if ASTERISK_VERSION_NUM < AST_4
		LOCAL_USER_REMOVE( u);
		#else	
		ast_module_user_remove(u);
		#endif
		return -1;
	}
	
	if(( argc = ast_app_separate_args( args, '|', argv, sizeof(argv) / sizeof(argv[0])))) 
	{
		text_or_file = argv[0];
		options = argv[1];
	}
		
	/* --------------------------------- */
	/* ----- Get options from user ----- */
	/* --------------------------------- */
	/* <init sil> */
	option_init_sil = strsep(&options, "|");
	if( option_init_sil)
	{
		if( !strcmp( option_init_sil, ""))
			option_init_sil = NULL;
	}
	/* <max sil> */
	option_max_sil = strsep(&options, "|");
	if( option_max_sil)
	{
		if( !strcmp( option_max_sil, ""))
			option_max_sil = NULL;
	}
	/* <TTS lang> */
	option_tts_lang = strsep(&options, "|");
	if( option_tts_lang)
	{
		if( !strcmp( option_tts_lang, ""))
			option_tts_lang = NULL;
	}
	/* <TTS spkr> */
	option_tts_spkr = strsep(&options, "|");
	if( option_tts_spkr)
	{
		if( !strcmp( option_tts_spkr, ""))
			option_tts_spkr = NULL;
	}
	/* <ASR config> */
	option_asr_config = strsep(&options, "|");
	if( option_asr_config)
	{
		if( !strcmp( option_asr_config, ""))
			option_asr_config = NULL;
	}
	/* <ASR lang> */
	option_asr_lang = strsep( &options, "|");
	if( option_asr_lang)
	{
		if( !strcmp( option_asr_lang, ""))
			option_asr_lang = NULL;
	}
	/* <Misc options> */
	opt = strsep(&options, "|");	
	if( opt) 
	{
		if( strchr( opt, 'f'))
			option_asfile = 1;/*tread text_or_file as a if it was a file*/
		if( strchr( opt, 'a'))
			option_answer = 1;
		if( strchr( opt, 'n'))
			option_no_hangup_on_verbio_err = 1;
		if( strchr( opt, 'b'))
			option_beep = 1;/*beep before rec */
		if( strchr( opt, 'g'))
		{
			option_bargein = 1;/*bargein activated */
			option_beep = 0;/* no sense if bargein active*/
		}
		if( strchr( opt, 'd'))
			option_dtmf = 1; /* enable dtmf detection*/	
		if( strchr( opt, 'i'))
		{
			option_interrupt = 1; /* immediate prompt interruption */
			option_bargein = 1; /* bargein needs to b active */	
		}
		if( strchr( opt, 'v'))
			option_verbose = 1;
	}
	
	/* ----------------------------------------------- */
	/* ----- Get options from Verbio config file ----- */
	/* ----------------------------------------------- */
	vcfg = ast_config_load( VERBIO_CFG);
	if( !vcfg) 
	{
		ast_log( LOG_ERROR, "Error opening configuration file %s\n", VERBIO_CFG);
		#if ASTERISK_VERSION_NUM < AST_4
		LOCAL_USER_REMOVE( u);
		#else	
		ast_module_user_remove(u);
		#endif

		return -1;
	}
	if( !( vcfg_primary_host = ast_variable_retrieve( vcfg, "general", "primary_vox_server"))) 
		vcfg_primary_host = VERBIO_DEF_HOST;
	
	if( !( vcfg_backup_host = ast_variable_retrieve( vcfg, "general", "backup_vox_server"))) 	
		vcfg_backup_host = VERBIO_DEF_HOST;
	
	if( !( vcfg_default_tts_lang = ast_variable_retrieve( vcfg, "tts", "default_language"))) 
		ast_log( LOG_WARNING, "Error reading default_language option\n");
	
	if( !( vcfg_default_tts_spkr = ast_variable_retrieve( vcfg, "tts", "default_speaker"))) 
		ast_log( LOG_WARNING, "Error reading default_speaker option\n");
	
	if( !( vcfg_tts_txt_path = ast_variable_retrieve( vcfg, "tts", "text_prompts_path"))) 
		ast_log( LOG_WARNING, "Error reading text_prompts_path option\n");
	
	if( !( tmp = ast_variable_retrieve( vcfg, "tts", "init_delay"))) 
		tmp = VERBIO_TTS_INIT_DELAY;
	vcfg_init_delay = atoi( tmp);
	
	if( !( tmp = ast_variable_retrieve( vcfg, "tts", "end_delay"))) 
		tmp = VERBIO_TTS_END_DELAY;
	vcfg_end_delay = atoi( tmp);
	
	if( !( tmp = ast_variable_retrieve(vcfg, "debug", "keep_synthesized_files"))) 
		tmp = "0";
	vcfg_keep_synthesized_files = atoi( tmp);
	
	if( !( vcfg_default_asr_config = ast_variable_retrieve(vcfg, "asr", "default_config"))) 
		ast_log( LOG_WARNING, "Error reading default_config option\n");

	if( !( vcfg_default_asr_lang = ast_variable_retrieve(vcfg, "asr", "default_language"))) 
		ast_log( LOG_WARNING, "Error reading default_language option\n");
	
	if( !( vcfg_init_sil = ast_variable_retrieve(vcfg, "asr", "init_sil"))) 
		vcfg_init_sil = VERBIO_INIT_SIL;

	if( !( vcfg_max_sil = ast_variable_retrieve(vcfg, "asr", "max_sil"))) 
		vcfg_max_sil = VERBIO_MAX_SIL;
	
	if( !( vcfg_max_ref = ast_variable_retrieve(vcfg, "asr", "max_ref"))) 
		vcfg_max_ref = NULL;
	
	if( !( vcfg_min_ref = ast_variable_retrieve(vcfg, "asr", "min_ref"))) 
		vcfg_min_ref = NULL;

	if( option_interrupt)
	{
		if( !( vcfg_low_factor = ast_variable_retrieve(vcfg, "vad", "low_factor"))) 
			vcfg_low_factor = NULL;
		
		if( !( vcfg_high_factor = ast_variable_retrieve(vcfg, "vad", "high_factor"))) 
			vcfg_high_factor = NULL;
		
		if( !( vcfg_final_factor = ast_variable_retrieve(vcfg, "vad", "final_factor"))) 
			vcfg_final_factor = NULL;
		
		if( !( vcfg_final_high_factor = ast_variable_retrieve(vcfg, "vad", "final_high_factor"))) 
			vcfg_final_high_factor = NULL;
		
		if( !( vcfg_min_high_thresh = ast_variable_retrieve(vcfg, "vad", "min_high_thresh"))) 
			vcfg_min_high_thresh = NULL;
		
		if( !( vcfg_aam_min = ast_variable_retrieve(vcfg, "vad", "aam_min"))) 
			vcfg_aam_min = NULL;
		
		if( !( vcfg_aam_max = ast_variable_retrieve(vcfg, "vad", "aam_max"))) 
			vcfg_aam_max = NULL;
	}

	if( !( tmp = ast_variable_retrieve(vcfg, "asr", "absolute_timeout"))) 
		tmp = VERBIO_DEF_ABS_TMOUT;
	vcfg_absolute_timeout = atoi( tmp);

	if( !( tmp = ast_variable_retrieve(vcfg, "debug", "keep_recorded_files"))) 
		tmp = "0";
	vcfg_keep_recorded_files = atoi( tmp);
	
	vcfg_recorded_files_path = ast_variable_retrieve(vcfg, "debug", "recorded_files_path");

	
	/* ---------------------------------------------------- */
	/* ----- Set options (to be used on voxlib calls) ----- */
	/* ---------------------------------------------------- */
	/* <init sil> */
	if( !option_init_sil)
		init_sil = atoi( vcfg_init_sil);
	else
		init_sil = atoi( option_init_sil);
	/* <max sil> */
	if( !option_max_sil)
		max_sil = atoi( vcfg_max_sil);
	else	
		max_sil = atoi( option_max_sil);
	/* <language> */
	if( !option_tts_lang)
		tts_lang = vcfg_default_tts_lang;
	else
		tts_lang = option_tts_lang;
	/* <Speaker> */
	if( !option_tts_spkr)
		tts_spkr = vcfg_default_tts_spkr;
	else	
		tts_spkr = option_tts_spkr;
	/* <Text to read> */
	if( option_asfile)
	{
		/* <Text file> */
		if( text_or_file[0] != '/')/*FIXME: elegance...*/
		{	/* relative path */
			strcpy( text_full_path, vcfg_tts_txt_path);
			strcat( text_full_path, "/");	
			strcat( text_full_path, text_or_file);	
		}
		else
		{	/* full path */
			strcpy( text_full_path, text_or_file);
		}
		text_to_synth = ast_read_textfile( text_full_path);
	}
	else
	{
		text_to_synth = text_or_file;
	}
	/* <ASR config> */
	if( !option_asr_config)
		asr_config = vcfg_default_asr_config;
	else
		asr_config = option_asr_config;
	/* <ASR lang> */
	if( !option_asr_lang)
		asr_lang = vcfg_default_asr_lang;
	else	
		asr_lang = option_asr_lang;
	/* <init sil> */
	if( !option_init_sil)
		init_sil = atoi( vcfg_init_sil);
	else
		init_sil = atoi( option_init_sil);
	/* <max sil> */
	if( !option_max_sil)
		max_sil = atoi( vcfg_max_sil);
	else	
		max_sil = atoi( option_max_sil);
	
	/* <dev value> */
	dev = verbio_get_dev( chan);
	
	if( option_interrupt)
	{
		if( vcfg_low_factor != NULL)
			low_factor = atof( vcfg_low_factor);
		else
			low_factor = VERBIO_LOW_FACTOR;
			
		if( vcfg_high_factor != NULL)
			high_factor = atof( vcfg_high_factor);
		else
			high_factor = VERBIO_HIGH_FACTOR;
	
		if( vcfg_final_factor != NULL)
			final_factor = atof( vcfg_final_factor);
		else
			final_factor = VERBIO_FINAL_FACTOR;
	
		if( vcfg_final_high_factor != NULL)
			final_high_factor = atof( vcfg_final_high_factor);
		else
			final_high_factor = VERBIO_FINAL_HIGH_FACTOR;

		if( vcfg_min_high_thresh != NULL)
			min_high_thresh = atof( vcfg_min_high_thresh);
		else
			min_high_thresh = VERBIO_MIN_HIGH_THRESH;

		if( vcfg_aam_min != NULL)
			aam_min = atof( vcfg_aam_min);
		else
			aam_min = VERBIO_AAM_MIN;

		if( vcfg_aam_max != NULL)
			aam_max = atof( vcfg_aam_max);
		else
			aam_max = VERBIO_AAM_MAX;
	}
	/* ---------------------------------------- */
	/* --- Save synthesized audio options  ---- */
	/* ---------------------------------------- */
	if( vcfg_keep_synthesized_files)
	{
		char t_str[VERBIO_MAX_TSTR_SIZE];
		time_t t = time( NULL );	
		//sprintf( t_str, "%d", (int)t);
		if( snprintf( t_str, VERBIO_MAX_TSTR_SIZE, "%d", (int)t) > VERBIO_MAX_TSTR_SIZE) 
			ast_log( LOG_WARNING, "Filename may be truncated.\n"); 
		strcat( filename_tts, t_str);
		strcat( filename_tts, "-");
		strcat( filename_tts, chan->uniqueid);
		if( vcfg_recorded_files_path)
		{
			strcpy( rec_tts_file_path, vcfg_recorded_files_path);
			strcat( rec_tts_file_path, "/");
			strcat( rec_tts_file_path, filename_tts);
		}
		else
		{
			strcpy( rec_tts_file_path, filename_tts);
		}
	}
	
	/* ---------------------------------------- */
	/* ----- Save recorded audio options  ----- */
	/* ---------------------------------------- */
	if( vcfg_keep_recorded_files)
	{
		char t_str[VERBIO_MAX_TSTR_SIZE];
		time_t t = time( NULL );	
		//sprintf( t_str, "%d", (int)t);
		if( snprintf( t_str, VERBIO_MAX_TSTR_SIZE, "%d", (int)t) > VERBIO_MAX_TSTR_SIZE) 
			ast_log( LOG_WARNING, "Filename may be truncated.\n"); 
		strcat(filename_asr, t_str);
		strcat(filename_asr, "-");
		strcat(filename_asr, chan->uniqueid);
		if( vcfg_recorded_files_path)
		{
			strcpy( rec_asr_file_path, vcfg_recorded_files_path);
			strcat( rec_asr_file_path, "/");
			strcat( rec_asr_file_path, filename_asr);
		}
		else
		{
			strcpy( rec_asr_file_path, filename_asr);
		}
	}
	/* <Print info to user...if (s)he wants> */
	if( option_verbose)
	{
		ast_log( LOG_NOTICE, "--------------------------\n");
		ast_log( LOG_NOTICE, "VerbioPrompt param summary:\n");
		ast_log( LOG_NOTICE, " Prim vox srv : %s\n", vcfg_primary_host);
		ast_log( LOG_NOTICE, " Bckp vox srv : %s\n", vcfg_backup_host);
		ast_log( LOG_NOTICE, " TTS language : %s\n", tts_lang);
		ast_log( LOG_NOTICE, " TTS speaker  : %s\n", tts_spkr);
		if( option_asfile)
			ast_log( LOG_NOTICE, " Synth from %s text file\n", text_full_path);
		ast_log( LOG_NOTICE, " Text to synth: %s\n", text_to_synth);
		ast_log( LOG_NOTICE, " Init delay   : %d\n", vcfg_init_delay);
		ast_log( LOG_NOTICE, " End delay    : %d\n", vcfg_end_delay);
		if( vcfg_keep_synthesized_files)
			ast_log( LOG_NOTICE, " Rec TTS file      :%s.%s\n", rec_tts_file_path, EXT);
		ast_log( LOG_NOTICE, " ASR config   : %s\n", asr_config);
		ast_log( LOG_NOTICE, " ASR lang     : %s\n", asr_lang);
		ast_log( LOG_NOTICE, " Init sil     : %d\n", init_sil);
		ast_log( LOG_NOTICE, " Max  sil     : %d\n", max_sil);
		ast_log( LOG_NOTICE, " Abs timeout  : %d\n", vcfg_absolute_timeout);
		if( vcfg_max_ref)
			ast_log( LOG_NOTICE, " Max  ref     : %d\n", atoi( vcfg_max_ref));
		if( vcfg_min_ref)
			ast_log( LOG_NOTICE, " Min  ref     : %d\n", atoi( vcfg_min_ref));
		if( option_interrupt)
		{
			ast_log( LOG_NOTICE, " Client side VAD options:\n");
			ast_log( LOG_NOTICE, "  low_factor\t: %f\n", low_factor);
			ast_log( LOG_NOTICE, "  high_factor\t: %f\n", high_factor);
			ast_log( LOG_NOTICE, "  final_factor\t: %f\n", final_factor);
			ast_log( LOG_NOTICE, "  final_high_factor: %f\n", final_high_factor);
			ast_log( LOG_NOTICE, "  min_high_thresh\t: %f\n", min_high_thresh);
			ast_log( LOG_NOTICE, "  aam_min\t: %f\n", aam_min);
			ast_log( LOG_NOTICE, "  aam_max\t: %f\n", aam_max);
		}
		if( vcfg_keep_recorded_files)
			ast_log( LOG_NOTICE, " Rec ASR file      :%s.%s\n", rec_asr_file_path, EXT);
		ast_log( LOG_NOTICE, " Voxlib device: %d\n", dev);
		ast_log( LOG_NOTICE, "--------------------------\n");
	}

	/* ----------------------------- */
	/* Answer the channel (if not up)*/
	/* ----------------------------- */
	int res = 0;
	if( chan->_state != AST_STATE_UP) 
	{
		if( option_answer) 
		{
			res = ast_answer(chan);
			
		} 
		else 
		{
			/* At the user's option, skip if the line is not up */
			#if ASTERISK_VERSION_NUM < AST_4
			LOCAL_USER_REMOVE( u);
			#else	
			ast_module_user_remove(u);
			#endif
			return -1;
		}
	}
	
	if( res) 
	{
		ast_log(LOG_WARNING, "Could not answer channel '%s'\n", chan->name);
		#if ASTERISK_VERSION_NUM < AST_4
		LOCAL_USER_REMOVE( u);
		#else	
		ast_module_user_remove(u);
		#endif
		return -1;
	}
	
	/* ------------------ */
	/* Set channel format */
	/* ------------------ */
	/* Verbio only accepts alaw ulaw and slin...we are going to use alaw*/
	if( ast_set_write_format( chan, AST_FORMAT_ALAW) <  0)
	{
		ast_log( LOG_NOTICE, "AST_FORMAT_ALAW (write) failed.\n");
		#if ASTERISK_VERSION_NUM < AST_4
		LOCAL_USER_REMOVE( u);
		#else	
		ast_module_user_remove(u);
		#endif

		return -1;
	}
	if( ast_set_read_format( chan, AST_FORMAT_ALAW) <  0)
	{
		ast_log( LOG_NOTICE, "AST_FORMAT_ALAW (read) failed.\n");
		#if ASTERISK_VERSION_NUM < AST_4
		LOCAL_USER_REMOVE( u);
		#else	
		ast_module_user_remove(u);
		#endif

		return -1;
	}
	
	/* ---------------------------------- */
	/* ----- Verbio Rec (ASR) stuff ----- */
	/* ---------------------------------- */
	vox_clrrsp( &rsp);
	rsp.maxsil = max_sil;
	rsp.initsil = init_sil;

	/* If we can not set asr config.. exit*/
	if( vox_setparm( dev,VXCH_DEFASRCFG, asr_config))
	{
		ast_log( LOG_ERROR, "Can not set ASR config to %s. [%s]\n", asr_config, ATVOX_ERRMSGP( dev));
		#if ASTERISK_VERSION_NUM < AST_4
		LOCAL_USER_REMOVE( u);
		#else	
		ast_module_user_remove(u);
		#endif
		if( option_no_hangup_on_verbio_err)
			return 0;
		else
			return -1;
	}
	/* If we can not set asr lang.. exit*/
	if( vox_setparm( dev,VXCH_DEFASRLNG, asr_lang))
	{
		ast_log( LOG_ERROR, "Can not set ASR lang to %s. [%s]\n", asr_lang, ATVOX_ERRMSGP( dev));
		#if ASTERISK_VERSION_NUM < AST_4
		LOCAL_USER_REMOVE( u);
		#else	
		ast_module_user_remove(u);
		#endif
		if( option_no_hangup_on_verbio_err)
			return 0;
		else
			return -1;
	}

	if( (recdev = vox_recstr_open( dev, &rsp, VERBIO_CODEC | MC_INITSIL )) == -1) 
	{
		ast_log(LOG_ERROR, "vox_recstr_open %ld;\n", ATVOX_LASTERR( dev));
		#if ASTERISK_VERSION_NUM < AST_4
		LOCAL_USER_REMOVE( u);
		#else	
		ast_module_user_remove(u);
		#endif
		if( option_no_hangup_on_verbio_err)
			return 0;
		else
			return -1;
	}

	/* maxref  VAD */
	if( vcfg_max_ref != NULL)
	{
		max_ref = atoi( vcfg_max_ref);
		if( vox_setparm( dev, VXGB_VSDMAXREF, &max_ref) == -1) 
			ast_log( LOG_ERROR, "Error setting max_ref.\n");
    	}
	/* minref  VAD */
	if( vcfg_min_ref != NULL)
	{
		if( vox_setparm( dev, VXGB_VSDMINREF, &min_ref) == -1)
			ast_log( LOG_ERROR, "Error setting min_ref.\n");	
	}

	/* keep a copy of what user says */
	if( vcfg_keep_recorded_files)
	{
		s_asr = ast_writefile( rec_asr_file_path, EXT, NULL, fflags , 0, 0644);
		/* Let the user know which file corresponds to actual recording*/
		pbx_builtin_setvar_helper( chan, "VASR_REC_FILE", rec_asr_file_path);
	}

	/* ---------------------------- */
	/* ----- Verbio VAD stuff ----- */
	/* ---------------------------- */
	if( option_interrupt)
	{
		VAD_PRM prm;
		vox_clrvad( &prm);
		prm.low_factor = low_factor;
		prm.high_factor = high_factor;
		prm.final_factor = final_factor;
		prm.final_high_factor = final_high_factor;
		prm.min_high_thresh = min_high_thresh;
		prm.aam_min = aam_min;
		prm.aam_max = aam_max;

		vsddev = vox_vsd_open( dev, recdev, &prm, VERBIO_CODEC);
		if( vsddev == -1)
		{
			ast_log(LOG_ERROR, "vsddev\n");
			vox_recstr_close( dev, recdev);
			#if ASTERISK_VERSION_NUM < AST_4
			LOCAL_USER_REMOVE( u);
			#else	
			ast_module_user_remove(u);
			#endif
			if( option_no_hangup_on_verbio_err)
				return 0;
			else
				return -1;
		}
	}

	/* ---------------------------- */
	/* ----- Verbio TTS stuff ----- */
	/* ---------------------------- */
	/* If we can not set tts lang.. exit*/
	if( vox_setparm( dev,VXCH_DEFTTSLANG, tts_lang))
	{
		ast_log( LOG_ERROR, "Can not set TTS language to %s. [%s]\n", tts_lang, ATVOX_ERRMSGP( dev));
		vox_recstr_close( dev, recdev);
		#if ASTERISK_VERSION_NUM < AST_4
		LOCAL_USER_REMOVE( u);
		#else	
		ast_module_user_remove(u);
		#endif
		if( option_no_hangup_on_verbio_err)
			return 0;
		else
			return -1;
	}
	/* Speaker is not critical */
	if( vox_setparm( dev,VXCH_TTSSPKNAME, tts_spkr))
		ast_log( LOG_WARNING, "Can not set TTS speaker to %s.\n", tts_spkr);
	
	
	/*Hardcoded..if we want to change any of these tts params...SSML tags*/
	/*Set volume*/
	int volume = 20000;
	vox_setparm(dev, VXCH_TTSVOLUME, &volume);
	
	/*Set speed*/
	int speed = 0; /* default*/
	vox_setparm(dev, VXCH_TTSSPEED, &speed);
	
	/*Set pitch*/
	int pitch = 0; /* default*/
	vox_setparm(dev, VXCH_TTSPITCH, &pitch);
	
	int synth = vox_playstr_open( dev, text_to_synth, VERBIO_CODEC);/*TODO:allow other formats? (ulaw and slin)*/
	if( !synth)
	{
		ast_log( LOG_ERROR, "Fatal error on playstr_open %s\n", ATVOX_ERRMSGP( dev));
		vox_recstr_close( dev, recdev);
		#if ASTERISK_VERSION_NUM < AST_4
		LOCAL_USER_REMOVE( u);
		#else	
		ast_module_user_remove(u);
		#endif
		if( option_no_hangup_on_verbio_err)
			return 0;
		else
			return -1;
	}

	/* keep a copy of synth audio */
	if( vcfg_keep_synthesized_files)
		s_tts = ast_writefile( rec_tts_file_path, EXT, NULL, fflags , 0, 0644);
	
	
	/* --------------------------------- */
	/* ----- Rec and Synth loop(s) ----- */
	/* --------------------------------- */
	/* Ensure no streams are currently running..FIXME:necessary? */
	ast_stopstream( chan);
	
	/*while( ast_waitfor( chan, -1) < -1)
		f = ast_read( chan);*/
	/* FIXME: First samples corrupted? */
	ast_safe_sleep( chan, vcfg_init_delay);
	int tts_has_finished = 0;/* Synth end check */
	
	/* 
	   If bargein option is not set, play our 
	   prompt (first) and then, recognize
	*/
	if( !option_bargein)
	{
		/* Synthesis loop */
		while( 1)
		{
			if( ast_waitfor( chan, -1) < 0)
			{
				ast_log(LOG_NOTICE, "Wait failed.\n");
				vox_playstr_close( dev, synth);
				vox_recstr_close( dev, recdev);
				if( option_interrupt)
					vox_vsd_close( dev, vsddev);
				ast_stopstream(chan);
				#if ASTERISK_VERSION_NUM < AST_4
				LOCAL_USER_REMOVE( u);
				#else	
				ast_module_user_remove(u);
				#endif
				return -1;	
			}
	
			/* Read a frame... we will reuse this frame for writing.
		   	   FIXME:create a new frame...?*/
			f = ast_read( chan);
			/* Hangup detection */
			if( !f)
			{
				ast_log(LOG_NOTICE, "Hangup detected.\n");
				vox_playstr_close( dev, synth);
				vox_recstr_close( dev, recdev);
				if( option_interrupt)
					vox_vsd_close( dev, vsddev);
				ast_stopstream(chan);
				#if ASTERISK_VERSION_NUM < AST_4
				LOCAL_USER_REMOVE( u);
				#else	
				ast_module_user_remove(u);
				#endif
				return -1;	
			}

			if( f->frametype == AST_FRAME_VOICE) 
			{
				/* reuse our audio_buff to store synth audio */
				/*TODO: avoid blocking functions */
				count = vox_playstr_read( synth, samples, f->samples);
				/* Set new samples in our frame */
				f->data = samples;
	
				if( vcfg_keep_synthesized_files)
				{
					/*Write frame to file*/
					if( ast_writestream( s_tts, f)) 
					{
						ast_log(LOG_WARNING, "Problem writing frame\n");
						ast_frfree(f);
						/*break;*/
					}
				}
				/*ast_queue_frame( chan, f);*//* internally frees f*/
				if( ast_write(chan, f))
				{ 
					/*TODO: handle this case*/
					ast_frfree(f);
					/*break;*/
				}
			}
			ast_frfree( f);
			
			/* Check if we are done */
			if( count < f->samples)
				break;
		}
		/* FIXME: Last samples (sometimes) corrupted? */
		ast_safe_sleep( chan, vcfg_end_delay);

		/* Clean  */
		vox_playstr_close( dev, synth);
		ast_stopstream(chan);
		count = 0;
	}

	/* Play a beep.*/
	if( option_beep)
	{
		/* Some code to play a nice little beep to signify the start of the record operation */
		if( !ast_streamfile( chan, "beep", chan->language)) 
			ast_waitstream( chan, "");
		else 
			ast_log( LOG_WARNING, "ast_streamfile failed on %s\n", chan->name);
		ast_stopstream( chan);
	}

	ast_stopstream( chan);

	/*
	The following loop will perform a recognition (if bargein is not active) 
	or a recognition and synthesis -at unison- (if bargein is active)
	*/	
	verbio_reset_dtmf_result( chan);/* Avoid confusion among recognitions*/
	int voice_detected = 0; /* voice detection flag */
	int sample_count = 0;
	while( 1)
	{
		if( ast_waitfor( chan, -1) < 0)
		{
			ast_log(LOG_NOTICE, "Wait failed.\n");
			vox_recstr_close( dev, recdev);
			vox_playstr_close( dev, synth);
			if( option_interrupt)
				vox_vsd_close( dev, vsddev);
			ast_stopstream(chan);
			#if ASTERISK_VERSION_NUM < AST_4
			LOCAL_USER_REMOVE( u);
			#else	
			ast_module_user_remove(u);
			#endif
			return -1;	
		}
		/* Read a frame from channel */
		f = ast_read( chan);

		if( !f)
		{
			ast_log(LOG_NOTICE, "Hangup detected.\n");
			vox_recstr_close( dev, recdev);
			vox_playstr_close( dev, synth);
			if( option_interrupt)
				vox_vsd_close( dev, vsddev);
			ast_stopstream(chan);
			#if ASTERISK_VERSION_NUM < AST_4
			LOCAL_USER_REMOVE( u);
			#else	
			ast_module_user_remove(u);
			#endif
			return -1;	
		}
	
		if( f->frametype == AST_FRAME_VOICE)
		{
			/* ----------------------- */
			/* ----- Recognition ----- */
			/* ----------------------- */
			if( vcfg_keep_recorded_files)
			{
				/*Write frame to file*/
				if( ast_writestream( s_asr, f)) 
				{
					ast_log(LOG_ERROR, "Problem writing frame\n");
					ast_frfree(f);
					/*break;*/
				}
			}
			
			if( option_interrupt && voice_detected != 1)
			{
				/* VAD Client-side */
				int status = 0;
				status = vox_vsd_write( vsddev, f->data, f->samples);
				if( (status != VVX_INIT) && (status != VVX_SILENCE))
				{
					voice_detected = 1;
					ast_log(LOG_NOTICE, "Voice detected\n");
				}

				/* VAD (Voice Activity Detection) Server-side*/
				/*vox_getparm( dev, VXCH_VOICEDETECTED, &voice_detected);*/
				/*if( voice_detected == 1)
					ast_stopstream( chan);*/
			}
			else 
			{
				/* Voice already detected */
				/*Pass samples to recognizer and check if we are done*/
				/*TODO: avoid blocking functions */
				if( vox_recstr_write( recdev, f->data, f->samples) < f->samples)
					break;/* Recognizer has something for us...
						Exit while loop and set result
						via verbio_set_asr_result function*/
			}	

			/* Absolute timeout counter */
			sample_count += f->samples;
			if( sample_count > (vcfg_absolute_timeout * 8000))
			{
				ast_log(LOG_WARNING, "Absolute timeout reached.\n");
				break; 
			}

			/* ------------------------------------- */
			/* ----- Synth (if bargein active) ----- */
			/* ------------------------------------- */
			if( option_bargein)
			{	
				if(( voice_detected != 1) && ( !tts_has_finished))
				{
					/* reuse our audio_buff to store synth audio */
					/*TODO: avoid blocking functions */
					count = vox_playstr_read( synth, samples, f->samples);
					/* Set new samples in our frame */
					f->data = samples;
		
					if( vcfg_keep_synthesized_files)
					{
						/*Write frame to file*/
						if( ast_writestream( s_tts, f)) 
						{
							ast_log(LOG_WARNING, "Problem writing frame\n");
							ast_frfree(f);
							/*break;*/
						}
					}

					if( ast_write(chan, f))
					{ 
						/*TODO: handle this case*/
						ast_frfree(f);
						/*break;*/
					}
				
					ast_frfree( f);
				}
				/* Check if TTS has finished */
				if( count < f->samples)
					tts_has_finished = 1;
			}
		}
		else if( f->frametype == AST_FRAME_DTMF) 
		{
			if( option_dtmf)
			{
				/* We are done ... set detected dtmf to dialplan var*/
				verbio_set_dtmf_result( chan, f);
				ast_log(LOG_NOTICE, "DTMF detected.\n");
				break; 
			}
		}
	}
	ast_stopstream(chan);

	/* Set (as channel vars) asr result(s) */
	verbio_set_asr_result( chan, dev);
	
	ast_config_destroy( vcfg);

	/*Voxlib cleanup*/
	vox_recstr_close( dev, recdev);
	vox_playstr_close( dev, synth);
	if( option_interrupt)
		vox_vsd_close( dev, vsddev);

	#if ASTERISK_VERSION_NUM < AST_4
	LOCAL_USER_REMOVE( u);
	#else	
	ast_module_user_remove(u);
	#endif

	
	return 0;	

}

/*! \brief Stream and recognition function (implements bargein). */
static int verbio_stream_and_rec( struct ast_channel *chan, void *data)
{
	/*
		This function allows us to launch a stream and a recognition at 
		the same time. As soon as the recognizer has some result for us,
		stream will be stoped (if bargein option is active).
	*/
	#if ASTERISK_VERSION_NUM < AST_4
	struct 			localuser *u;
	#else
	struct 			ast_module_user *u;
	#endif
	/* Input options */
	char 	*options	= NULL;
	char 	*audio_file 	= NULL;
	char 	*args;
	int 	argc = 0;
	char	*argv[2];
	char	*opt;
	/* -- */
	int 	option_answer 		= 0;
	int	option_no_hangup_on_verbio_err = 0;
	int 	option_beep 		= 0;
	int	option_bargein 		= 0;
	int 	option_interrupt	= 0;
	int 	option_verbose 		= 0;
	int 	option_dtmf		= 0;
	char	*option_init_sil	= NULL;
	char 	*option_max_sil 	= NULL;
	char	*option_asr_config 	= NULL;
	char	*option_asr_lang 	= NULL;
	/* Vars to store cfg file options*/
	const char 	*vcfg_primary_host	= NULL;
	const char 	*vcfg_backup_host	= NULL;
	int 		vcfg_keep_recorded_files= 0;
	const char 	*vcfg_default_asr_config= NULL;
	const char 	*vcfg_default_asr_lang 	= NULL;
	const char 	*vcfg_max_sil 		= NULL;
	const char 	*vcfg_init_sil		= NULL;
	const char	*vcfg_recorded_files_path = NULL;
	const char	*vcfg_max_ref		= NULL;	
	const char 	*vcfg_min_ref		= NULL;
	int 		vcfg_absolute_timeout 	= 0;
	/* VAD Client-side params*/
	const char	*vcfg_low_factor	= NULL;
	const char	*vcfg_high_factor	= NULL;
	const char	*vcfg_final_factor	= NULL;
	const char	*vcfg_final_high_factor	= NULL;
	const char	*vcfg_min_high_thresh	= NULL;
	const char	*vcfg_aam_min		= NULL;
	const char	*vcfg_aam_max		= NULL;
	/* Options to be used when dealing with voxlib*/
	int 		dev 		= 0;
	const char	*asr_config 	= NULL;
	const char	*asr_lang 	= NULL;
	int 		recdev = 0;
	int 		vsddev = 0;
	VX_RSP 	rsp;
	int 	init_sil= 300;
	int 	max_sil	= 200;
	int	max_ref	= 0;
	int 	min_ref	= 0;
	float 	low_factor = 0.0;
	float 	high_factor = 0.0;
	float 	final_factor = 0.0;
	float 	final_high_factor = 0.0;
	float 	min_high_thresh = 0.0;
	float 	aam_min = 0.0;
	float 	aam_max = 0.0;

	/* Keep files */
	const int 	fflags 			= O_CREAT|O_TRUNC|O_WRONLY;
	struct 		ast_filestream *s_asr 	= '\0';
	char 		filename_asr[MAXPATHLEN] 	= "verbio-rec-";
	char 		rec_asr_file_path[MAXPATHLEN];
	/* Helper vars*/
	struct 		ast_frame *f;
	char *front = NULL;
	/*char *back = NULL;*/
	const char 	*tmp;
	struct 		ast_config 	*vcfg;
	
	
	if( ast_strlen_zero(data)) 
	{
		ast_log( LOG_WARNING, "%s requires an argument (audio_file[|initsil][|maxsil][|asr_conf][|asr_lang][|options])\n",verbio_stream_and_rec_app);
		return -1;
	}
	
	#if ASTERISK_VERSION_NUM < AST_4
	LOCAL_USER_ADD( u);
	#else
	u = ast_module_user_add(chan);
	#endif
	
	/* We need to make a copy of the input string if we are going to modify it! */
	args = ast_strdupa( data);	
	if( !args) 
	{
		ast_log( LOG_ERROR, "Out of memory!\n");
		#if ASTERISK_VERSION_NUM < AST_4
		LOCAL_USER_REMOVE( u);
		#else	
		ast_module_user_remove(u);
		#endif
		return -1;
	}
	
	if(( argc = ast_app_separate_args( args, '|', argv, sizeof(argv) / sizeof(argv[0])))) 
	{
		audio_file = argv[0];
		options = argv[1];
	}
		
	/* --------------------------------- */
	/* ----- Get options from user ----- */
	/* --------------------------------- */
	/* <init sil> */
	option_init_sil = strsep(&options, "|");
	if( option_init_sil)
	{
		if( !strcmp( option_init_sil, ""))
			option_init_sil = NULL;
	}
	/* <max sil> */
	option_max_sil = strsep(&options, "|");
	if( option_max_sil)
	{
		if( !strcmp( option_max_sil, ""))
			option_max_sil = NULL;
	}
	/* <ASR config> */
	option_asr_config = strsep(&options, "|");
	if( option_asr_config)
	{
		if( !strcmp( option_asr_config, ""))
			option_asr_config = NULL;
	}
	/* <ASR lang> */
	option_asr_lang = strsep( &options, "|");
	if( option_asr_lang)
	{
		if( !strcmp( option_asr_lang, ""))
			option_asr_lang = NULL;
	}
	/* <Misc options> */
	opt = strsep(&options, "|");	
	if( opt) 
	{
		if( strchr( opt, 'a'))
			option_answer = 1;
		if( strchr( opt, 'n'))
			option_no_hangup_on_verbio_err = 1;
		if( strchr( opt, 'b'))
			option_beep = 1;/*beep before rec */
		if( strchr( opt, 'g'))
		{
			option_bargein = 1;/*bargein activated */
			option_beep = 0;/* no sense if bargein active*/
		}
		if( strchr( opt, 'd'))
			option_dtmf = 1; /* enable dtmf detection*/	
		if( strchr( opt, 'i'))
		{
			option_interrupt = 1; /* immediate prompt interruption */
			option_bargein = 1; /* bargein needs to b active */	
		}
		if( strchr( opt, 'v'))
			option_verbose = 1;
	}
	
	/* ----------------------------------------------- */
	/* ----- Get options from Verbio config file ----- */
	/* ----------------------------------------------- */
	vcfg = ast_config_load( VERBIO_CFG);
	if( !vcfg) 
	{
		ast_log( LOG_ERROR, "Error opening configuration file %s\n", VERBIO_CFG);
		#if ASTERISK_VERSION_NUM < AST_4
		LOCAL_USER_REMOVE( u);
		#else	
		ast_module_user_remove(u);
		#endif
		return -1;
	}
	if( !( vcfg_primary_host = ast_variable_retrieve( vcfg, "general", "primary_vox_server"))) 
		vcfg_primary_host = VERBIO_DEF_HOST;
	
	if( !( vcfg_backup_host = ast_variable_retrieve( vcfg, "general", "backup_vox_server"))) 	
		vcfg_backup_host = VERBIO_DEF_HOST;
	
	if( !( vcfg_default_asr_config = ast_variable_retrieve(vcfg, "asr", "default_config"))) 
		ast_log( LOG_WARNING, "Error reading default_config option\n");

	if( !( vcfg_default_asr_lang = ast_variable_retrieve(vcfg, "asr", "default_language"))) 
		ast_log( LOG_WARNING, "Error reading default_language option\n");
	
	if( !( vcfg_init_sil = ast_variable_retrieve(vcfg, "asr", "init_sil"))) 
		vcfg_init_sil = VERBIO_INIT_SIL;

	if( !( vcfg_max_sil = ast_variable_retrieve(vcfg, "asr", "max_sil"))) 
		vcfg_max_sil = VERBIO_MAX_SIL;
	
	if( !( vcfg_max_ref = ast_variable_retrieve(vcfg, "asr", "max_ref"))) 
		vcfg_max_ref = NULL;
	
	if( !( vcfg_min_ref = ast_variable_retrieve(vcfg, "asr", "min_ref"))) 
		vcfg_min_ref = NULL;

	if( option_interrupt)
	{
		if( !( vcfg_low_factor = ast_variable_retrieve(vcfg, "vad", "low_factor"))) 
			vcfg_low_factor = NULL;
		
		if( !( vcfg_high_factor = ast_variable_retrieve(vcfg, "vad", "high_factor"))) 
			vcfg_high_factor = NULL;
		
		if( !( vcfg_final_factor = ast_variable_retrieve(vcfg, "vad", "final_factor"))) 
			vcfg_final_factor = NULL;
		
		if( !( vcfg_final_high_factor = ast_variable_retrieve(vcfg, "vad", "final_high_factor"))) 
			vcfg_final_high_factor = NULL;
		
		if( !( vcfg_min_high_thresh = ast_variable_retrieve(vcfg, "vad", "min_high_thresh"))) 
			vcfg_min_high_thresh = NULL;
		
		if( !( vcfg_aam_min = ast_variable_retrieve(vcfg, "vad", "aam_min"))) 
			vcfg_aam_min = NULL;
		
		if( !( vcfg_aam_max = ast_variable_retrieve(vcfg, "vad", "aam_max"))) 
			vcfg_aam_max = NULL;
	}

	if( !( tmp = ast_variable_retrieve(vcfg, "asr", "absolute_timeout"))) 
		tmp = VERBIO_DEF_ABS_TMOUT;
	vcfg_absolute_timeout = atoi( tmp);
	
	if( !( tmp = ast_variable_retrieve(vcfg, "debug", "keep_recorded_files"))) 
		tmp = "0";
	vcfg_keep_recorded_files = atoi( tmp);
	
	vcfg_recorded_files_path = ast_variable_retrieve(vcfg, "debug", "recorded_files_path");

	
	/* ---------------------------------------------------- */
	/* ----- Set options (to be used on voxlib calls) ----- */
	/* ---------------------------------------------------- */
	/* <init sil> */
	if( !option_init_sil)
		init_sil = atoi( vcfg_init_sil);
	else
		init_sil = atoi( option_init_sil);
	/* <max sil> */
	if( !option_max_sil)
		max_sil = atoi( vcfg_max_sil);
	else	
		max_sil = atoi( option_max_sil);
	/* <ASR config> */
	if( !option_asr_config)
		asr_config = vcfg_default_asr_config;
	else
		asr_config = option_asr_config;
	/* <ASR lang> */
	if( !option_asr_lang)
		asr_lang = vcfg_default_asr_lang;
	else	
		asr_lang = option_asr_lang;
	/* <init sil> */
	if( !option_init_sil)
		init_sil = atoi( vcfg_init_sil);
	else
		init_sil = atoi( option_init_sil);
	/* <max sil> */
	if( !option_max_sil)
		max_sil = atoi( vcfg_max_sil);
	else	
		max_sil = atoi( option_max_sil);
	
	/* <dev value> */
	dev = verbio_get_dev( chan);

	/* Client-side VAD options */
	if( option_interrupt)
	{
		if( vcfg_low_factor != NULL)
			low_factor = atof( vcfg_low_factor);
		else
			low_factor = VERBIO_LOW_FACTOR;
			
		if( vcfg_high_factor != NULL)
			high_factor = atof( vcfg_high_factor);
		else
			high_factor = VERBIO_HIGH_FACTOR;
	
		if( vcfg_final_factor != NULL)
			final_factor = atof( vcfg_final_factor);
		else
			final_factor = VERBIO_FINAL_FACTOR;
	
		if( vcfg_final_high_factor != NULL)
			final_high_factor = atof( vcfg_final_high_factor);
		else
			final_high_factor = VERBIO_FINAL_HIGH_FACTOR;

		if( vcfg_min_high_thresh != NULL)
			min_high_thresh = atof( vcfg_min_high_thresh);
		else
			min_high_thresh = VERBIO_MIN_HIGH_THRESH;

		if( vcfg_aam_min != NULL)
			aam_min = atof( vcfg_aam_min);
		else
			aam_min = VERBIO_AAM_MIN;

		if( vcfg_aam_max != NULL)
			aam_max = atof( vcfg_aam_max);
		else
			aam_max = VERBIO_AAM_MAX;
	}
	
	/* ---------------------------------------- */
	/* ----- Save recorded audio options  ----- */
	/* ---------------------------------------- */
	if( vcfg_keep_recorded_files)
	{
		char t_str[VERBIO_MAX_TSTR_SIZE];
		time_t t = time( NULL );	
		//sprintf( t_str, "%d", (int)t);
		if( snprintf( t_str, VERBIO_MAX_TSTR_SIZE, "%d", (int)t) > VERBIO_MAX_TSTR_SIZE) 
			ast_log( LOG_WARNING, "Filename may be truncated.\n"); 
		strcat(filename_asr, t_str);
		strcat(filename_asr, "-");
		strcat(filename_asr, chan->uniqueid);
		if( vcfg_recorded_files_path)
		{
			strcpy( rec_asr_file_path, vcfg_recorded_files_path);
			strcat( rec_asr_file_path, "/");
			strcat( rec_asr_file_path, filename_asr);
		}
		else
		{
			strcpy( rec_asr_file_path, filename_asr);
		}
	}
	/* <Print info to user...if (s)he wants> */
	if( option_verbose)
	{
		ast_log( LOG_NOTICE, "---------------------------------\n");
		ast_log( LOG_NOTICE, "VerbioStreamAndRec param summary:\n");
		ast_log( LOG_NOTICE, " Prim vox srv : %s\n", vcfg_primary_host);
		ast_log( LOG_NOTICE, " Bckp vox srv : %s\n", vcfg_backup_host);
		ast_log( LOG_NOTICE, " File to play : %s\n", audio_file);
		ast_log( LOG_NOTICE, " ASR config   : %s\n", asr_config);
		ast_log( LOG_NOTICE, " ASR lang     : %s\n", asr_lang);
		ast_log( LOG_NOTICE, " Init sil     : %d\n", init_sil);
		ast_log( LOG_NOTICE, " Max  sil     : %d\n", max_sil);
		ast_log( LOG_NOTICE, " Abs timeout  : %d\n", vcfg_absolute_timeout);
		if( vcfg_max_ref)
			ast_log( LOG_NOTICE, " Max  ref     : %d\n", atoi( vcfg_max_ref));
		if( vcfg_min_ref)
			ast_log( LOG_NOTICE, " Min  ref     : %d\n", atoi( vcfg_min_ref));
		if( option_interrupt)
		{
			ast_log( LOG_NOTICE, " Client side VAD options:\n");
			ast_log( LOG_NOTICE, "  low_factor\t: %f\n", low_factor);
			ast_log( LOG_NOTICE, "  high_factor\t: %f\n", high_factor);
			ast_log( LOG_NOTICE, "  final_factor\t: %f\n", final_factor);
			ast_log( LOG_NOTICE, "  final_high_factor: %f\n", final_high_factor);
			ast_log( LOG_NOTICE, "  min_high_thresh\t: %f\n", min_high_thresh);
			ast_log( LOG_NOTICE, "  aam_min\t: %f\n", aam_min);
			ast_log( LOG_NOTICE, "  aam_max\t: %f\n", aam_max);
		}
		if( vcfg_keep_recorded_files)
			ast_log( LOG_NOTICE, " Rec ASR file      :%s.%s\n", rec_asr_file_path, EXT);
		ast_log( LOG_NOTICE, " Voxlib device: %d\n", dev);
		ast_log( LOG_NOTICE, "---------------------------------\n");
	}

	/* ----------------------------- */
	/* Answer the channel (if not up)*/
	/* ----------------------------- */
	int res = 0;
	if( chan->_state != AST_STATE_UP) 
	{
		if( option_answer) 
		{
			res = ast_answer(chan);
			
		} 
		else 
		{
			/* At the user's option, skip if the line is not up */
			#if ASTERISK_VERSION_NUM < AST_4
			LOCAL_USER_REMOVE( u);
			#else	
			ast_module_user_remove(u);
			#endif
			return -1;
		}
	}
	
	if( res) 
	{
		ast_log(LOG_WARNING, "Could not answer channel '%s'\n", chan->name);
		#if ASTERISK_VERSION_NUM < AST_4
		LOCAL_USER_REMOVE( u);
		#else	
		ast_module_user_remove(u);
		#endif
		return -1;
	}
	
	/* ------------------ */
	/* Set channel format */
	/* ------------------ */
	/* Verbio only accepts alaw ulaw and slin...*/
	if( ast_set_write_format( chan, AST_FORMAT_ALAW) <  0)
	{
		ast_log( LOG_NOTICE, "AST_FORMAT_ALAW (write) failed.\n");
		#if ASTERISK_VERSION_NUM < AST_4
		LOCAL_USER_REMOVE( u);
		#else	
		ast_module_user_remove(u);
		#endif

		return -1;
	}
	if( ast_set_read_format( chan, AST_FORMAT_ALAW) <  0)
	{
		ast_log( LOG_NOTICE, "AST_FORMAT_ALAW (read) failed.\n");
		#if ASTERISK_VERSION_NUM < AST_4
		LOCAL_USER_REMOVE( u);
		#else	
		ast_module_user_remove(u);
		#endif

		return -1;
	}
	
	/* ---------------------------------- */
	/* ----- Verbio Rec (ASR) stuff ----- */
	/* ---------------------------------- */
	vox_clrrsp( &rsp);
	rsp.maxsil = max_sil;
	rsp.initsil = init_sil;

	/* If we can not set asr config.. exit*/
	if( vox_setparm( dev,VXCH_DEFASRCFG, asr_config))
	{
		ast_log( LOG_ERROR, "Can not set ASR config to %s. [%s]\n", asr_config, ATVOX_ERRMSGP( dev));
		#if ASTERISK_VERSION_NUM < AST_4
		LOCAL_USER_REMOVE( u);
		#else	
		ast_module_user_remove(u);
		#endif
		if( option_no_hangup_on_verbio_err)
			return 0;
		else
			return -1;
	}
	/* If we can not set asr lang.. exit*/
	if( vox_setparm( dev,VXCH_DEFASRLNG, asr_lang))
	{
		ast_log( LOG_ERROR, "Can not set ASR lang to %s. [%s]\n", asr_lang, ATVOX_ERRMSGP( dev));
		#if ASTERISK_VERSION_NUM < AST_4
		LOCAL_USER_REMOVE( u);
		#else	
		ast_module_user_remove(u);
		#endif
		if( option_no_hangup_on_verbio_err)
			return 0;
		else
			return -1;
	}

	if( (recdev = vox_recstr_open( dev, &rsp, VERBIO_CODEC | MC_INITSIL )) == -1) 
	{
		ast_log(LOG_ERROR, "vox_recstr_open %d. [%s]\n", dev, ATVOX_ERRMSGP( dev));
		#if ASTERISK_VERSION_NUM < AST_4
		LOCAL_USER_REMOVE( u);
		#else	
		ast_module_user_remove(u);
		#endif
		if( option_no_hangup_on_verbio_err)
			return 0;
		else
			return -1;
	}

	/* maxref  VAD */
	if( vcfg_max_ref != NULL)
	{
		max_ref = atoi( vcfg_max_ref);
		if( vox_setparm( dev, VXGB_VSDMAXREF, &max_ref) == -1) 
			ast_log( LOG_ERROR, "Error setting max_ref.\n");
    	}
	/* minref  VAD */
	if( vcfg_min_ref != NULL)
	{
		if( vox_setparm( dev, VXGB_VSDMINREF, &min_ref) == -1)
			ast_log( LOG_ERROR, "Error setting min_ref.\n");	
	}

	/* keep a copy of what user says */
	if( vcfg_keep_recorded_files)
	{
		s_asr = ast_writefile( rec_asr_file_path, EXT, NULL, fflags , 0, 0644);
		/* Let the user know which file corresponds to actual recording*/
		pbx_builtin_setvar_helper( chan, "VASR_REC_FILE", rec_asr_file_path);
	}
	
	/* ---------------------------- */
	/* ----- Verbio VAD stuff ----- */
	/* ---------------------------- */
	if( option_interrupt)
	{
		VAD_PRM prm;
		vox_clrvad( &prm);
		prm.low_factor = low_factor;
		prm.high_factor = high_factor;
		prm.final_factor = final_factor;
		prm.final_high_factor = final_high_factor;
		prm.min_high_thresh = min_high_thresh;
		prm.aam_min = aam_min;
		prm.aam_max = aam_max;

		vsddev = vox_vsd_open( dev, recdev, &prm, VERBIO_CODEC);
		if( vsddev == -1)
		{
			ast_log(LOG_ERROR, "vsddev\n");
			vox_recstr_close( dev, recdev);
			#if ASTERISK_VERSION_NUM < AST_4
			LOCAL_USER_REMOVE( u);
			#else	
			ast_module_user_remove(u);
			#endif
			if( option_no_hangup_on_verbio_err)
				return 0;
			else
				return -1;
		}
	}

	/* ---------------------------------- */
	/* ----- Rec and stream loop(s) ----- */
	/* ---------------------------------- */
	/* Ensure no streams are currently running..FIXME:necessary? */
	ast_stopstream( chan);
	
	/* Stream file */
	front = audio_file;
	res = ast_streamfile( chan, front, chan->language);
	if( res)
	{
		ast_log(LOG_WARNING, "ast_streamfile failed on %s for %s\n", chan->name, (char *)front);
		vox_recstr_close( dev, recdev);
		return -1;
	}
	
	/*while( !res && front != NULL) 
	{
		if(( back = strchr(front, '&'))) 
		{
			*back = '\0';
			back++;
		}
		res = ast_streamfile( chan, front, chan->language);
		if( res) 
		{ 
			ast_log(LOG_WARNING, "ast_streamfile failed on %s for %s\n", chan->name, (char *)data);
			vox_recstr_close( dev, recdev);
			return 0;
		}
		front = back;
	}*/

	if( !option_bargein)
	{
		/* Wait for audio stream to finish */
		res = ast_waitstream(chan, "");	
		if( res)
		{
			ast_log(LOG_WARNING, "ast_waitstream failed on %s \n", chan->name);
			vox_recstr_close( dev, recdev);
			if( option_interrupt)
				vox_vsd_close( dev, vsddev);
			return -1;	
		}
		
		if( option_beep)
		{
			/* Some code to play a nice little beep to signify the start of the record operation */
			if( !ast_streamfile( chan, "beep", chan->language)) 
				ast_waitstream( chan, "");
			else 
				ast_log( LOG_WARNING, "ast_streamfile failed on %s\n", chan->name);
		}
		ast_stopstream(chan);
	}

	/* Recognition loop */
	verbio_reset_dtmf_result( chan);/* Avoid confusion among recognitions*/
	int voice_detected = 0; /*voice detection flag*/
	int sample_count = 0;
	while( 1)
	{
		if( option_bargein)
			res = ast_sched_wait(chan->sched);
		else
			res = -1;

		if( ast_waitfor( chan, res) < 0)
		{
			ast_log(LOG_ERROR, "Wait failed.\n");
			vox_recstr_close( dev, recdev);
			if( option_interrupt)
				vox_vsd_close( dev, vsddev);
			ast_stopstream(chan);
			#if ASTERISK_VERSION_NUM < AST_4
			LOCAL_USER_REMOVE( u);
			#else	
			ast_module_user_remove(u);
			#endif
			return -1;
		} 

		f = ast_read(chan);
		if( !f) 
		{
			ast_log( LOG_NOTICE, "Hangup detected.\n");
			vox_recstr_close( dev, recdev);
			if( option_interrupt)
				vox_vsd_close( dev, vsddev);
			ast_stopstream(chan);
			#if ASTERISK_VERSION_NUM < AST_4
			LOCAL_USER_REMOVE( u);
			#else	
			ast_module_user_remove(u);
			#endif
			return -1;
		} 
	
		if(( f->frametype == AST_FRAME_VOICE) && ( f->subclass == AST_FORMAT_ALAW)) 
		{
			/* Write frame to file */
			if( vcfg_keep_recorded_files)
			{
				if( ast_writestream( s_asr, f)) 
				{
					ast_log(LOG_WARNING, "Problem writing frame\n");
					ast_frfree(f);
					/*break;*/
				}
			}

			if( option_interrupt && voice_detected != 1)
			{
				int status = 0;
				status = vox_vsd_write( vsddev, f->data, f->samples);
				if( (status != VVX_INIT) && (status != VVX_SILENCE))
				{
					voice_detected = 1;
					ast_stopstream( chan);
					ast_log(LOG_NOTICE, "Voice detected\n");
				}
				/* do VAD (Voice Activity Detection) */
				/*vox_getparm( dev, VXCH_VOICEDETECTED, &voice_detected);*/
				/*if( voice_detected == 1)
				ast_stopstream( chan);*/
			}
			else 
			{
				/* Voice already detected */
				/*Pass samples to recognizer and check if we are done*/
				/*TODO: avoid blocking functions */
				if( vox_recstr_write( recdev, f->data, f->samples) < f->samples)
					break;/* Recognizer has something for us...
						Exit while loop and set result
						via verbio_set_asr_result function*/
			}
			
			/* Absolute timeout counter */
			sample_count += f->samples;
			if( sample_count > (vcfg_absolute_timeout * 8000))
			{
				ast_log(LOG_WARNING, "Absolute timeout reached.\n");
				break; 
			}
		}
		else if( f->frametype == AST_FRAME_DTMF) 
		{
			if( option_dtmf)
			{
				/* We are done ... set detected dtmf to dialplan var*/
				verbio_set_dtmf_result( chan, f);
				ast_log(LOG_NOTICE, "DTMF detected.\n");
				break; 
			}
		}
		ast_frfree( f);
		
		if( option_bargein)
			ast_sched_runq(chan->sched);
	}
	ast_stopstream( chan);

	/* Set (as channel vars) asr result(s) */
	verbio_set_asr_result( chan, dev);

	ast_config_destroy( vcfg);

	/* Voxlib cleanup */
	vox_recstr_close( dev, recdev);
	if( option_interrupt)
		vox_vsd_close( dev, vsddev);
	
	#if ASTERISK_VERSION_NUM < AST_4
	LOCAL_USER_REMOVE( u);
	#else	
	ast_module_user_remove(u);
	#endif

	
	return 0;	

}

/*! \brief Load vocabulary. This app must be called prior to any recognition. */
static int verbio_load_vcb( struct ast_channel *chan, void *data)
{
	#if ASTERISK_VERSION_NUM < AST_4
	struct 			localuser *u;
	#else
	struct 			ast_module_user *u;
	#endif
	/* Options */
	char 		*args;
	int 		argc = 0;
	char 		*argv[3];
	struct ast_config *vcfg;
	char		*options 		= NULL;
	char		*opt			= NULL;
	int		option_no_hangup_on_verbio_err = 0;
	int 		option_verbose 		= 0;
	char		*option_asr_config 	= NULL;
	char		*option_asr_lang 	= NULL;
	const char 	*vcfg_default_asr_config= NULL;
	const char 	*vcfg_default_asr_lang 	= NULL;
	/* Options to be used when dealing with voxlib */
	const char 	*vcfg_primary_host	= NULL;
	const char 	*vcfg_backup_host	= NULL;
	int 		dev 		= 0;
	char 		gram_full_path[MAXPATHLEN];
	const char	*asr_config 	= NULL;
	const char	*asr_lang 	= NULL;
	int		nind		= 0;
	int 		setmode 	= GVX_ISOLATED;   /*GVX_ABNF/GVX_ISOLATED*/
	int 		vc_handle 	= 0; /* Vocabulary id */
	/* Misc vars */
	const char 	*grammar_path	= NULL;
	char 		*gram_file	= NULL;
	char 		*gram_type 	= NULL;
	

	if( ast_strlen_zero(data)) 
	{
		ast_log(LOG_WARNING, "%s requires an argument (gram_file|gram_type|[config]|[lang]|[options])\n",verbio_load_vcb_app);
		return -1;
	}

	#if ASTERISK_VERSION_NUM < AST_4
	LOCAL_USER_ADD( u);
	#else
	u = ast_module_user_add(chan);
	#endif
	
	/* We need to make a copy of the input string if we are going to modify it! */
	args = ast_strdupa(data);	
	if( !args) 
	{
		ast_log(LOG_ERROR, "Out of memory!\n");
		#if ASTERISK_VERSION_NUM < AST_4
		LOCAL_USER_REMOVE( u);
		#else	
		ast_module_user_remove(u);
		#endif
		return -1;
	}

	if(( argc = ast_app_separate_args(args, '|', argv, sizeof(argv) / sizeof(argv[0])))) 
	{
		gram_file = argv[0];
		gram_type = argv[1];
		options	  = argv[2];
	}

	if( ast_strlen_zero( gram_file) && ast_strlen_zero( gram_type)) 
	{
		ast_log(LOG_ERROR, "Check input parameters.\n");
		#if ASTERISK_VERSION_NUM < AST_4
		LOCAL_USER_REMOVE( u);
		#else	
		ast_module_user_remove(u);
		#endif
		return -1;

	}	

	/* --------------------------------- */
	/* ----- Get options from user ----- */
	/* --------------------------------- */
	/* <ASR config> */
	option_asr_config = strsep(&options, "|");
	if( option_asr_config)
	{
		if( !strcmp( option_asr_config, ""))
			option_asr_config = NULL;
	}
	/* <ASR lang> */
	option_asr_lang = strsep(&options, "|");
	if( option_asr_lang)
	{
		if( !strcmp( option_asr_lang, ""))
			option_asr_lang = NULL;
	}
	/* <Misc options> */
	opt = strsep(&options, "|");	
	if( opt) 
	{
		if( strchr( opt, 'v'))
			option_verbose = 1;
		if( strchr( opt, 'n'))
			option_no_hangup_on_verbio_err = 1;
	}


	/* ----------------------------------------------- */
	/* ----- Get options from Verbio config file ----- */
	/* ----------------------------------------------- */
	vcfg = ast_config_load( VERBIO_CFG);
	if( !vcfg) 
	{
		ast_log(LOG_WARNING, "Error opening configuration file %s\n", VERBIO_CFG);
		#if ASTERISK_VERSION_NUM < AST_4
		LOCAL_USER_REMOVE( u);
		#else	
		ast_module_user_remove(u);
		#endif
		return -1;
	}
	if( !( vcfg_primary_host = ast_variable_retrieve(vcfg, "general", "primary_vox_server"))) 
		vcfg_primary_host = VERBIO_DEF_HOST;

	if( !( vcfg_backup_host = ast_variable_retrieve(vcfg, "general", "backup_vox_server"))) 	
		vcfg_backup_host = VERBIO_DEF_HOST;
	
	if( !( vcfg_default_asr_config = ast_variable_retrieve(vcfg, "asr", "default_config"))) 
		ast_log( LOG_WARNING, "Error reading default_config option\n");

	if( !( vcfg_default_asr_lang = ast_variable_retrieve(vcfg, "asr", "default_language"))) 
		ast_log( LOG_WARNING, "Error reading default_language option\n");

	if( !( grammar_path = ast_variable_retrieve(vcfg, "asr", "grammar_path"))) 
		ast_log( LOG_WARNING, "Error reading grammar_path option\n");

	/* ---------------------------------------------------- */
	/* ----- Set options (to be used on voxlib calls) ----- */
	/* ---------------------------------------------------- */
	/* <ASR config> */
	if( !option_asr_config)
		asr_config = vcfg_default_asr_config;
	else
		asr_config = option_asr_config;
	/* <ASR lang> */
	if( !option_asr_lang)
		asr_lang = vcfg_default_asr_lang;
	else	
		asr_lang = option_asr_lang;
	/* <Grammar file> */
	if( gram_file[0] != '/')
	{	
		strcpy( gram_full_path, grammar_path);
		strcat( gram_full_path, "/");	
		strcat( gram_full_path, gram_file);	
	}
	else
	{
		strcpy( gram_full_path, gram_file);
	}
	/* <Grammar type ISOLATED or ABNF> */	
	/*TODO: be more 'elegant'...*/ 
	if( toupper( gram_type[0]) == 'I')
		setmode = GVX_ISOLATED;
	if( toupper( gram_type[0]) == 'A')
		setmode = GVX_ABNF;
	
	/* <dev value> */
	dev = verbio_get_dev( chan);

	/* <Print info to user...if (s)he wants> */
	if( option_verbose)
	{
		ast_log( LOG_NOTICE, "--------------------------\n");
		ast_log( LOG_NOTICE, "VerbioLoadVcb param summary:\n");
		ast_log( LOG_NOTICE, " Prim vox srv : %s\n", vcfg_primary_host);
		ast_log( LOG_NOTICE, " Bckp vox srv : %s\n", vcfg_backup_host);
		ast_log( LOG_NOTICE, " Gram path    : %s\n", gram_full_path);
		ast_log( LOG_NOTICE, " ASR config   : %s\n", asr_config);
		ast_log( LOG_NOTICE, " ASR lang     : %s\n", asr_lang);
		if( setmode == GVX_ISOLATED)
			ast_log(LOG_NOTICE, " Grammar type: ISOLATED\n");
		if( setmode == GVX_ABNF)
			ast_log(LOG_NOTICE, " Grammar type: ABNF\n");
		ast_log( LOG_NOTICE, " Voxlib device: %d\n", dev);
		ast_log( LOG_NOTICE, "--------------------------\n");
	}

	/* ---------------------------- */
	/* ----- Verbio Vcb stuff ----- */
	/* ---------------------------- */

	/* FIXME: voxlib bug that does not allow us to change 
	   	  change ASR cfg and LANG on a specified dev.
		  If we always use the same cfg... no problem.
	*/
	/* If we can not set asr config.. exit*/
	if( vox_setparm( dev,VXCH_DEFASRCFG, asr_config) == -1)
	{
		ast_log( LOG_ERROR, "Can not set ASR config to %s. [%s]\n", asr_config, ATVOX_ERRMSGP( dev));
		#if ASTERISK_VERSION_NUM < AST_4
		LOCAL_USER_REMOVE( u);
		#else	
		ast_module_user_remove(u);
		#endif
		if( option_no_hangup_on_verbio_err)
			return 0;
		else
			return -1;
	}
	
	/* If we can not set asr lang.. exit*/
	if( vox_setparm( dev,VXCH_DEFASRLANG, asr_lang) == -1)
	{
		ast_log( LOG_ERROR, "Can not set ASR lang to %s. [%s]\n", asr_lang, ATVOX_ERRMSGP( dev));
		#if ASTERISK_VERSION_NUM < AST_4
		LOCAL_USER_REMOVE( u);
		#else	
		ast_module_user_remove(u);
		#endif
		if( option_no_hangup_on_verbio_err)
			return 0;
		else
			return -1;
	}
	
	/* Prepare vocabulary */
	if( vox_prevcbex( gram_full_path, setmode, &nind) == -1) 
	{
		
		ast_log(LOG_ERROR, "vox_prevcbex [%s line: %d]\n", ATVOX_ERRMSGP( -1), nind);
		#if ASTERISK_VERSION_NUM < AST_4
		LOCAL_USER_REMOVE( u);
		#else	
		ast_module_user_remove(u);
		#endif
		if( option_no_hangup_on_verbio_err)
			return 0;
		else
			return -1;
	}
	
	/* Load vocabulary */
	if( (vc_handle = vox_loadvcb( dev, gram_full_path, setmode)) < 0) 
	{
		ast_log(LOG_ERROR, "vox_loadvcb [%s]\n", ATVOX_ERRMSGP( dev));
		#if ASTERISK_VERSION_NUM < AST_4
		LOCAL_USER_REMOVE( u);
		#else	
		ast_module_user_remove(u);
		#endif
		if( option_no_hangup_on_verbio_err)
			return 0;
		else
			return -1;
	}
	
	/* Activate vocabulary */
	if( vox_activatevcb( dev, vc_handle, setmode) < 0)
	{
		ast_log(LOG_ERROR, "vox_activatevcb [%s]\n", ATVOX_ERRMSGP( dev));
		#if ASTERISK_VERSION_NUM < AST_4
		LOCAL_USER_REMOVE( u);
		#else	
		ast_module_user_remove(u);
		#endif
		if( option_no_hangup_on_verbio_err)
			return 0;
		else
			return -1;
	}

	/* ---------------------------- */
	/* ----- Set vcb chan var ----- */
	/* ---------------------------- */
	char tmpbuff [VERBIO_MAX_INT_SIZE];
  	/*int size;*/
	/*This (Asterisk) channel var will allow us to reference
	this vocabulary (i.e on vocabulary unload)
	*/
	//size = sprintf( tmpbuff, "%d", vc_handle);
	/*sprintf not safe (used snprintf...C99?)*/

	if( snprintf( tmpbuff, VERBIO_MAX_INT_SIZE, "%d",  nind) > VERBIO_MAX_INT_SIZE)
		ast_log( LOG_WARNING, "VVCB_HANDLE may be truncated.\n");
  	pbx_builtin_setvar_helper( chan, "VVCB_HANDLE", tmpbuff);

	if( option_verbose)
		ast_log(LOG_NOTICE, "vc_handle %d\n", vc_handle);

	ast_config_destroy( vcfg);

	#if ASTERISK_VERSION_NUM < AST_4
	LOCAL_USER_REMOVE( u);
	#else	
	ast_module_user_remove(u);
	#endif

	return 0;
}

/*! \brief Unload vocabulary. */
static int verbio_unload_vcb( struct ast_channel *chan, void *data)
{
	int 		res = 0;
	#if ASTERISK_VERSION_NUM < AST_4
	struct 			localuser *u;
	#else
	struct 			ast_module_user *u;
	#endif
	int 		argc = 0;
	char 		*argv[2];
	struct ast_config *vcfg;
	const char 	*tmp= NULL;
	char 		*args = NULL;
	/* Input Options */
	char		*options 	= NULL;
	char		*option_asr_config = NULL;
	char 		*option_asr_lang = NULL;
	char		*opt 	= NULL;
	int		option_no_hangup_on_verbio_err = 0;
	int 		option_verbose	= 0;
	/* Cfg file options */
	const char 	*vcfg_default_asr_config = NULL;
	const char 	*vcfg_default_asr_lang = NULL;
	/* Options to be used when dealing with voxlib */
	const char 	*vcfg_primary_host	= NULL;
	const char 	*vcfg_backup_host	= NULL;
	int 		dev 		= 0;
	const char	*asr_config 	= NULL;
	const char	*asr_lang 	= NULL;
	int		vcb_handle	= 0;
	

	if( ast_strlen_zero( data)) 
	{
		ast_log( LOG_WARNING, "%s requires arguments (vcbhandle|[config]|[lang])\n",verbio_unload_vcb_app);
		return -1;
	}

	#if ASTERISK_VERSION_NUM < AST_4
	LOCAL_USER_ADD( u);
	#else
	u = ast_module_user_add( chan);
	#endif
	
	args = ast_strdupa( data);	
	if( !args) 
	{
		ast_log(LOG_ERROR, "Out of memory!\n");
		#if ASTERISK_VERSION_NUM < AST_4
		LOCAL_USER_REMOVE( u);
		#else	
		ast_module_user_remove(u);
		#endif
		return -1;
	}

	if(( argc = ast_app_separate_args(args, '|', argv, sizeof(argv) / sizeof(argv[0])))) 
	{
		tmp = argv[0];
		options = argv[1];
	}

 	if( !ast_strlen_zero( tmp))
	{
		vcb_handle = atoi( tmp);
	}
	else
	{
		ast_log(LOG_ERROR, "Check input parameters.\n");
		#if ASTERISK_VERSION_NUM < AST_4
		LOCAL_USER_REMOVE( u);
		#else	
		ast_module_user_remove(u);
		#endif

		return -1;
	}	

	/* --------------------------------- */
	/* ----- Get options from user ----- */
	/* --------------------------------- */
	/* <ASR config> */
	option_asr_config = strsep(&options, "|");
	if( option_asr_config)
	{
		if( !strcmp( option_asr_config, ""))
			option_asr_config = NULL;
	}
	/* <ASR lang> */
	option_asr_lang = strsep(&options, "|");
	if( option_asr_lang)
	{
		if( !strcmp( option_asr_lang, ""))
			option_asr_lang = NULL;
	}
	/* <Misc options> */
	opt = strsep(&options, "|");	
	if( opt) 
	{
		if( strchr( opt, 'n'))
			option_no_hangup_on_verbio_err = 1;
		if( strchr( opt, 'v'))
			option_verbose = 1;
	}

	/* ----------------------------------------------- */
	/* ----- Get options from Verbio config file ----- */
	/* ----------------------------------------------- */
	vcfg = ast_config_load( VERBIO_CFG);
	if( !vcfg) 
	{
		ast_log(LOG_WARNING, "Error opening configuration file %s\n", VERBIO_CFG);
		#if ASTERISK_VERSION_NUM < AST_4
		LOCAL_USER_REMOVE( u);
		#else	
		ast_module_user_remove(u);
		#endif
		return -1;
	}
	if( !( vcfg_primary_host = ast_variable_retrieve(vcfg, "general", "primary_vox_server"))) 
		vcfg_primary_host = VERBIO_DEF_HOST;
	
	if( !( vcfg_backup_host = ast_variable_retrieve(vcfg, "general", "backup_vox_server"))) 	
		vcfg_backup_host = VERBIO_DEF_HOST;
	
	if( !( vcfg_default_asr_config = ast_variable_retrieve(vcfg, "asr", "default_config"))) 
		ast_log( LOG_WARNING, "Error reading default_config option\n");

	if( !( vcfg_default_asr_lang = ast_variable_retrieve(vcfg, "asr", "default_language"))) 
		ast_log( LOG_WARNING, "Error reading default_language option\n");
		

	/* ---------------------------------------------------- */
	/* ----- Set options (to be used on voxlib calls) ----- */
	/* ---------------------------------------------------- */
	/* <ASR config> */
	if( !option_asr_config)
		asr_config = vcfg_default_asr_config;
	else
		asr_config = option_asr_config;
	/* <ASR lang> */
	if( !option_asr_lang)
		asr_lang = vcfg_default_asr_lang;
	else	
		asr_lang = option_asr_lang;
	/* <dev value> */
	dev = verbio_get_dev( chan);
		
	/* <Print info to user...if (s)he wants> */
	if( option_verbose)
	{
		ast_log( LOG_NOTICE, "------------------------------\n");
		ast_log( LOG_NOTICE, "VerbioUnloadVcb param summary:\n");
		ast_log( LOG_NOTICE, " Prim vox srv : %s\n", vcfg_primary_host);
		ast_log( LOG_NOTICE, " Bckp vox srv : %s\n", vcfg_backup_host);
		ast_log( LOG_NOTICE, " ASR config   : %s\n", asr_config);
		ast_log( LOG_NOTICE, " ASR lang     :   %s\n", asr_lang);
		ast_log( LOG_NOTICE, " Voxlib device: %d\n", dev);
		ast_log( LOG_NOTICE, "------------------------------\n");
	}
		
	/* ---------------------------- */
	/* ----- Verbio uVcb stuff ---- */
	/* ---------------------------- */
	/* We do not need all this code...CFG and LNG previously set*/
	/* If we can not set asr config.. exit*/
	/*if( vox_setparm( dev,VXCH_DEFASRCFG, asr_config))
	{
		ast_log( LOG_ERROR, "Can not set ASR config to %s. [%s]\n", asr_config, ATVOX_ERRMSGP( dev));
		#if ASTERISK_VERSION_NUM < AST_4
		LOCAL_USER_REMOVE( u);
		#else	
		ast_module_user_remove(u);
		#endif
		if( option_no_hangup_on_verbio_err)
			return 0;
		else
			return -1;
	}*/
	/* If we can not set asr lang.. exit*/
	/*if( vox_setparm( dev,VXCH_DEFASRLNG, asr_lang))
	{
		ast_log( LOG_ERROR, "Can not set ASR lang to %s. [%s]\n", asr_lang, ATVOX_ERRMSGP( dev));
		#if ASTERISK_VERSION_NUM < AST_4
		LOCAL_USER_REMOVE( u);
		#else	
		ast_module_user_remove(u);
		#endif
		if( option_no_hangup_on_verbio_err)
			return 0;
		else
			return -1;
	}*/
	
	if( vcb_handle == -1)
	{
		if( vox_setvcb( dev, NULL, 0))/* Free licences */
			ast_log( LOG_ERROR, "vox_setvcb. [%s]\n", ATVOX_ERRMSGP( dev));
		
		vox_devclose( dev); /* Free dev resources...otherwise we will run out of lines*/
	}
	else if( vox_unloadvcb( dev, vcb_handle, 0) < 0) 
	{
		ast_log( LOG_ERROR, "vox_unloadvcb. [%s]\n", ATVOX_ERRMSGP( dev));
		#if ASTERISK_VERSION_NUM < AST_4
		LOCAL_USER_REMOVE( u);
		#else	
		ast_module_user_remove(u);
		#endif
		if( option_no_hangup_on_verbio_err)
			return 0;
		else
			return -1;
	}

	ast_config_destroy( vcfg);
	
	#if ASTERISK_VERSION_NUM < AST_4
	LOCAL_USER_REMOVE( u);
	#else	
	ast_module_user_remove(u);
	#endif

	return res;
}

static int verbio_info( struct ast_channel *chan, void *data)
{
	/*
		Testing purposes function.
	*/
	#if ASTERISK_VERSION_NUM < AST_4
	struct 			localuser *u;
	#else
	struct 			ast_module_user *u;
	#endif
	struct ast_config *vcfg;
	const char *primary_host;
	const char *backup_host;
	const char *vcfg_net_timeout;
	const char *tmp;
	const char *default_tts_lang;
	const char *default_tts_spkr;
	const char *tts_txt_path;
	const char *default_asr_config;
	const char *default_asr_lang;
	const char *asr_gram_path;
	int max_sil;

	#if ASTERISK_VERSION_NUM < AST_4
	LOCAL_USER_ADD( u);
	#else
	u = ast_module_user_add( chan);
	#endif
	
	vcfg = ast_config_load( VERBIO_CFG);
	if( !vcfg) 
	{
		ast_log( LOG_WARNING, "Error opening configuration file %s\n", VERBIO_CFG);
		#if ASTERISK_VERSION_NUM < AST_4
		LOCAL_USER_REMOVE( u);
		#else	
		ast_module_user_remove( u);
		#endif
		return -1;
	}
	if( !( primary_host = ast_variable_retrieve( vcfg, "general", "primary_vox_server"))) 
		primary_host = VERBIO_DEF_HOST;
	ast_log( LOG_NOTICE, "Primary Host: %s\n", primary_host);
	
	if( !( backup_host = ast_variable_retrieve( vcfg, "general", "backup_vox_server"))) 	
		backup_host = VERBIO_DEF_HOST; 
	ast_log( LOG_NOTICE, "Backup Host: %s\n", backup_host);

	if( !( vcfg_net_timeout = ast_variable_retrieve( vcfg, "general", "net_timeout"))) 
		vcfg_net_timeout = VERBIO_DEF_NET_TMOUT;
	
	ast_log( LOG_NOTICE, "Net timeout: %s\n", vcfg_net_timeout);

	if( !( default_tts_lang = ast_variable_retrieve( vcfg, "tts", "default_language"))) 
		default_tts_lang = "NOT_SET";
	ast_log( LOG_NOTICE, "Default TTS lang: %s\n", default_tts_lang);
	ast_log( LOG_NOTICE, "\t - licenses %d\n", vox_getttslic( default_tts_lang));

	if( !( default_tts_spkr = ast_variable_retrieve( vcfg, "tts", "default_speaker"))) 
		default_tts_spkr = "NOT_SET";
	ast_log( LOG_NOTICE, "Default TTS speaker: %s\n", default_tts_spkr);
	
	if( !( tts_txt_path = ast_variable_retrieve( vcfg, "tts", "text_prompts_path"))) 
		tts_txt_path = "NOT_SET";
	ast_log( LOG_NOTICE, "Default TTS txt path: %s\n", tts_txt_path);
	
	if( !( default_asr_config = ast_variable_retrieve( vcfg, "asr", "default_config"))) 
		default_asr_config = "NOT_SET";
	ast_log( LOG_NOTICE, "Default ASR config: %s\n", default_asr_config);
	ast_log( LOG_NOTICE, "\t - licenses %d\n", vox_getasrlic( default_asr_config));
	
	if( !( default_asr_lang = ast_variable_retrieve( vcfg, "asr", "default_language"))) 
		default_asr_lang = "NOT_SET";
	ast_log( LOG_NOTICE, "Default ASR lang: %s\n", default_asr_lang);

	if( !( asr_gram_path = ast_variable_retrieve( vcfg, "asr", "grammar_path"))) 
		asr_gram_path = "NOT_SET";
	ast_log( LOG_NOTICE, "Default ASR grammar path: %s\n", asr_gram_path);

	if( !( tmp = ast_variable_retrieve( vcfg, "asr", "max_sil"))) 
		tmp = "100";
	max_sil = atoi( tmp);
	ast_log( LOG_NOTICE, "Max Sil: %d\n", max_sil);

	ast_config_destroy( vcfg);

	/* ----------------------------- */
	/* Answer the channel (if not up)*/
	/* ----------------------------- */
	int res = 0;
	if( chan->_state != AST_STATE_UP) 
		res = ast_answer( chan);
	
	if( res) 
	{
		ast_log( LOG_WARNING, "Could not answer channel '%s'\n", chan->name);
		#if ASTERISK_VERSION_NUM < AST_4
		LOCAL_USER_REMOVE( u);
		#else	
		ast_module_user_remove( u);
		#endif
		return -1;
	}

	/* Print samples per frame info */
	struct ast_frame *f;
	int n = 0;
	do
	{	if( ast_waitfor( chan, -1) > -1)
		{
			f = ast_read( chan);
			if( !f)
			{
				ast_log( LOG_NOTICE, "Hangup\n");
				#if ASTERISK_VERSION_NUM < AST_4
				LOCAL_USER_REMOVE( u);
				#else	
				ast_module_user_remove( u);
				#endif
				return -1;
			}
			ast_log( LOG_NOTICE, "Samples per frame(%d): %d\n", n, f->samples);
		}
		++n;
	}while( n < 4);

	#if ASTERISK_VERSION_NUM < AST_4
	LOCAL_USER_REMOVE( u);
	#else	
	ast_module_user_remove( u);
	#endif


	return 0;
}

static int verbio_last_err( struct ast_channel *chan, void *data)
{
	/*
		This function will check if there's any Verbio error
		(will set an Asterisk chan var describing last occurred error) 
	*/
	#if ASTERISK_VERSION_NUM < AST_4
	struct 			localuser *u;
	#else
	struct 			ast_module_user *u;
	#endif
	/* Input options */
	char 			*var = NULL;
	char 			*args;
	int 			argc = 0;
	char			*argv[1];
	
	if( ast_strlen_zero( data)) 
	{
		ast_log( LOG_ERROR, "%s requires an argument (var)\n",verbio_last_err_app);
		return -1;
	}

	#if ASTERISK_VERSION_NUM < AST_4
	LOCAL_USER_ADD( u);
	#else
	u = ast_module_user_add( chan);
	#endif

	/* We need to make a copy of the input string if we are going to modify it! */
	args = ast_strdupa( data);	
	if( !args) 
	{
		ast_log( LOG_ERROR, "Out of memory!\n");
		#if ASTERISK_VERSION_NUM < AST_4
		LOCAL_USER_REMOVE( u);
		#else	
		ast_module_user_remove(u);
		#endif
		return -1;
	}
	
	if(( argc = ast_app_separate_args( args, '|', argv, sizeof(argv) / sizeof(argv[0])))) 
		var = argv[0];

	verbio_set_err( chan, verbio_get_dev( chan), var);

	#if ASTERISK_VERSION_NUM < AST_4
	LOCAL_USER_REMOVE( u);
	#else	
	ast_module_user_remove( u);
	#endif


	return 0;
}

#if ASTERISK_VERSION_NUM < AST_4
int unload_module( void)
#else	
static int unload_module( void)
#endif
{
	int res = 0;

	ast_unregister_application( verbio_prompt_app);
	ast_unregister_application( verbio_rec_app);
	ast_unregister_application( verbio_load_vcb_app);
	ast_unregister_application( verbio_unload_vcb_app);
	ast_unregister_application( verbio_prompt_and_rec_app);
	ast_unregister_application( verbio_stream_and_rec_app);

	res = ast_unregister_application( verbio_info_app);
	res |= ast_unregister_application( verbio_last_err_app);

	#if ASTERISK_VERSION_NUM < AST_4
	STANDARD_HANGUP_LOCALUSERS;
	#else	
	ast_module_user_hangup_all();
	#endif

	/* Close voxlib */
	ast_log( LOG_NOTICE, "Closing Verbio voxlib.\n");
	vox_libclose();
	sleep( 1);	
	
	return res;	
}

#if ASTERISK_VERSION_NUM < AST_4
int load_module( void)
#else	
static int load_module( void)
#endif
{
	int res = 0;
	/* Vars to store cfg file options*/
	struct ast_config 	*vcfg;	
	const char 		*vcfg_primary_host;
	const char 		*vcfg_backup_host;
	const char 		*vcfg_net_timeout;
	const char 		*vcfg_default_asr_config;
	const char 		*vcfg_default_asr_lang;
	const char 		*vcfg_default_tts_lang;
	const char 		*vcfg_default_tts_spkr;
	/* ASR and TTS availability */
	int			asr = 0;
	int			tts = 0;
	const unsigned int	retries = 10;/*TODO: to config file*/
	unsigned int		done = 0;
	unsigned int		count = 0;
	
	/* ----------------------------------------------- */
	/* ----- Get options from Verbio config file ----- */
	/* ----------------------------------------------- */
	vcfg = ast_config_load( VERBIO_CFG);
	
	if( !vcfg) 
	{
		ast_log(LOG_ERROR, "Error opening configuration file %s\n", VERBIO_CFG);
		return -1;
	}
	if( !( vcfg_primary_host = ast_variable_retrieve(vcfg, "general", "primary_vox_server"))) 
		vcfg_primary_host = VERBIO_DEF_HOST;
	
	if( !( vcfg_backup_host = ast_variable_retrieve(vcfg, "general", "backup_vox_server"))) 	
		vcfg_backup_host = VERBIO_DEF_HOST;
	
	if( !( vcfg_net_timeout = ast_variable_retrieve(vcfg, "general", "net_timeout"))) 
		vcfg_net_timeout = VERBIO_DEF_NET_TMOUT;

	if( !( vcfg_default_asr_config = ast_variable_retrieve(vcfg, "asr", "default_config"))) 
		ast_log( LOG_WARNING, "ASR 'default_config' option missing...ASR will not be available.\n");

	if( !( vcfg_default_asr_lang = ast_variable_retrieve(vcfg, "asr", "default_language"))) 
		ast_log( LOG_WARNING, "ASR 'default_language' option missing...ASR will not be available.\n");
		
	if( !( vcfg_default_tts_lang = ast_variable_retrieve( vcfg, "tts", "default_language"))) 
		ast_log( LOG_WARNING, "TTS 'default_language' option missing...TTS will not be available.\n");
	
	if( !( vcfg_default_tts_spkr = ast_variable_retrieve( vcfg, "tts", "default_speaker"))) 
		ast_log( LOG_WARNING, "TTS default_speaker missing.\n");

	/* If we perform a 'restart now' from CLI unload_module won't get called.
	   We need to disconnect prior to reconnect to verbio voxserver.
	*/
	vox_libclose();
	
	/* ---------------------------- */
	/* ----- Verbio Init stuff ---- */
	/* ---------------------------- */
	/* Connect on load_module...we avoid reconnections.
	   Vox server periodically sends messages to clients.
	*/
	/* Set primary vox server ip*/
	vox_setparm( -1, VXGB_DEFSERVER, vcfg_primary_host);
	/* Set timeout */
	vox_setparm( -1, VXGB_NETTIMEOUT, vcfg_net_timeout);
	
 	/* Init VOX library */
	/* Try to init ASR*/
	if( vcfg_default_asr_config && vcfg_default_asr_lang)
	{
		done = 0;
		count = 0;
		ast_log( LOG_NOTICE, "Starting ASR engine...\n");
		while( (count < retries) && (done == 0))
		{
			asr = 1;/*assume that all will run ok, optimism is the way*/
			sleep( 1);
			if( vox_asr_init( vcfg_default_asr_config, vcfg_default_asr_lang) == -1) 
			{
				/*failed...try to connect to backup host*/
				/*ast_log( LOG_WARNING, "Can not init ASR on %s trying %s\n", vcfg_primary_host, vcfg_backup_host);
				ast_log( LOG_WARNING, "ERROR: %s\n",ATVOX_ERRMSGP( -1));*/
				vox_setparm( -1, VXGB_DEFSERVER, vcfg_backup_host);
				if( vox_asr_init( vcfg_default_asr_config, vcfg_default_asr_lang) == -1) 
					asr = 0;/*murphy*/
				else
					done = 1;
			}
			else
			{
				done = 1;
			}
			++count;
		}
		if( done == 1)
			ast_log( LOG_NOTICE, "success.\n");
		else //count == retries
			ast_log( LOG_ERROR, "Can not init ASR (tried primary [%s] and backup [%s] hosts). %s\n", vcfg_primary_host, vcfg_backup_host, ATVOX_ERRMSGP( -1));
	}
	
	/* Try to init TTS */
	if( vcfg_default_tts_lang)
	{
		done = 0;
		count = 0;
		ast_log( LOG_NOTICE, "Starting TTS engine...\n");
		while( (count < retries) && (done == 0))
		{
			tts = 1;
			sleep( 1);
			if( vox_tts_init( NULL, vcfg_default_tts_lang) == -1) 
			{
				/*failed...try to connect to backup host*/
				/*ast_log( LOG_WARNING, "Can not init TTS on %s trying %s", vcfg_primary_host, vcfg_backup_host);*/
				vox_setparm( -1, VXGB_DEFSERVER, vcfg_backup_host);
				if( vox_tts_init( NULL, vcfg_default_tts_lang) == -1) 
					tts = 0;
				else
					done = 1;
			}
			else
			{
				done = 1;
			}
			++count;
		}
		if( done == 1)
			ast_log( LOG_NOTICE, "success.\n");
		else /* count == retries */
			ast_log( LOG_ERROR, "Can not init TTS (tried primary [%s] and backup [%s] hosts). %s\n", vcfg_primary_host, vcfg_backup_host, ATVOX_ERRMSGP( -1));
	}

	/* Load applications that are not voxserver dependant */
	res = ast_register_application( verbio_info_app, verbio_info, "Print some (maybe useful) Info", verbio_info_descrip);
	res |= ast_register_application( verbio_last_err_app, verbio_last_err, "Get Verbio last error", verbio_last_err_descrip);
	
	if( asr)
	{
		/* maxref  VAD */
		vox_setparm( -1, VXGB_VSDMAXREF, &VERBIO_MAX_REF); 

		/* Register ASR related apps */
		res |= ast_register_application( verbio_load_vcb_app, verbio_load_vcb, "Load vocabulary", verbio_load_vcb_descrip);
		res |= ast_register_application( verbio_unload_vcb_app, verbio_unload_vcb, "Unload vocabulary", verbio_unload_vcb_descrip);
		res |= ast_register_application( verbio_rec_app, verbio_rec, "Recognize application", verbio_rec_descrip);
		res |= ast_register_application( verbio_stream_and_rec_app, verbio_stream_and_rec, "Stream and recognize application", verbio_stream_and_rec_descrip);
	}
	if( tts)
	{
		/* Register TTS related apps */
		res  |= ast_register_application( verbio_prompt_app, verbio_prompt, "Text to speech application", verbio_prompt_descrip);
		
	}

	if( asr && tts)
	{
		/* Register ASR and TTS related apps */
		res  |= ast_register_application( verbio_prompt_and_rec_app, verbio_prompt_and_rec, "Launch synthesis and recognition", verbio_prompt_and_rec_descrip);
	}	

	ast_config_destroy( vcfg);

	return res;
}
 
#if ASTERISK_VERSION_NUM < AST_4
int usecount( void)
{
 	int res;
 	STANDARD_USECOUNT(res);
 	return res;
}

char *key()
{
	return ASTERISK_GPL_KEY;
}

char *description( void)
{
 	return tdesc;
}
#endif

#if ASTERISK_VERSION_NUM < AST_4
/*AST_MODULE_INFO_STANDARD(ASTERISK_GPL_KEY, "Verbio Speech Technologies Applications");*/
#else
AST_MODULE_INFO(ASTERISK_GPL_KEY, AST_MODFLAG_DEFAULT, "Verbio Speech Technologies Applications",
		.load = load_module,
		.unload = unload_module,
	       );
#endif

