/*
 * Asterisk -- An open source telephony toolkit.
 *
 * Copyright (C) 2008, Verbio Technologies S.L.
 *
 * Verbio Technologies <support@verbio.com>
 *
 * See http://www.asterisk.org for more information about
 * the Asterisk project. Please do not directly contact
 * any of the maintainers of this project for assistance;
 * the project provides a web site, mailing lists and IRC
 * channels for your use.
 *
 * This program is free software, distributed under the terms of
 * the GNU General Public License Version 2. See the LICENSE file
 * at the top of the source tree.
 */

/*! \file
 *
 * \brief Verbio Speech Technologies connector
 *
 * \author Verbio Technologies S.L <support@verbio.com>
 */

#include "asterisk.h"

#include <stdio.h>
#include <stdlib.h>
#include <unistd.h>
#include <string.h>
#include <errno.h>

#include "asterisk/file.h"
#include "asterisk/channel.h"
#include "asterisk/module.h"
#include "asterisk/lock.h"
#include "asterisk/linkedlists.h"
#include "asterisk/cli.h"
#include "asterisk/term.h"
#include "asterisk/options.h"
#include "asterisk/speech.h"
#include "asterisk/version.h"

/* Verbio voxlib */
#include <voxlib.h> 	/*We also need to link against libvoxlib.
			  voxlib.h and libvoxlib.so are included in 
			  verbio-clients package.
			*/

/* For Asterisk VideoCaps */
#if ASTERISK_VERSION_NUM == 999999 
 #undef ASTERISK_VERSION_NUM
 #define ASTERISK_VERSION_NUM 10499
#endif

#define	VERBIO_CODEC 	MC_LIN16

static const char	*VERBIO_CFG		= "verbio.conf";
static const char	*VERBIO_DEF_HOST	= "127.0.0.1";
static const int 	VERBIO_MAXINDEX 	= 32;
static const int 	VERBIO_MAX_RES_SIZE	= 100;
static const int 	VERBIO_MAX_CVAR_SIZE	= 120;
static const int 	VERBIO_MAX_INT_SIZE	= 10;
static const float 	VERBIO_INIT_SIL 	= 300.0;
static const float 	VERBIO_MAX_SIL 		= 200.0;
static const float 	VERBIO_MAX_REF		= 200.0;
static const float 	VERBIO_MIN_REF		= 50.0;
static const int 	VERBIO_DEF_ABS_TIMEOUT	= 30;
static const float	VERBIO_LOW_FACTOR	= 2.5;
static const float	VERBIO_HIGH_FACTOR	= 4.5;
static const float	VERBIO_FINAL_FACTOR	= 0.8;
static const float	VERBIO_FINAL_HIGH_FACTOR= 2.5;
static const float	VERBIO_MIN_HIGH_THRESH	= 500.0;
static const float	VERBIO_AAM_MIN		= 50.0;
static const float	VERBIO_AAM_MAX		= 200.0;
static const float 	VSTD_FACTOR		= 2.0;
static const float 	VN_FACTOR		= 4.0;
static const float 	VH_FACTOR		= 6.0;

static const float	VERBIO_MAX_DEV		= 800000;
static const int 	VERBIO_MAX_GRAM_NUM	= 100;

static const char	*base_fname	= "rsv-";
static const char	*exten		= "pcm";

static const char	*VERBIO_GRM_CACHE_DIR	= "/.cache/";

#define	VERBIO_DIR_MODE		0777
#define	VERBIO_FILE_MODE	0644

#define AST_4 			10400
#define AST_6 			10600

typedef enum {VVAD_INIT_SIL,
		VVAD_MAX_SIL,
		VVAD_MAX_REF,
		VVAD_MIN_REF,
		VVAD_LOW_FACTOR,
		VVAD_HIGH_FACTOR,
		VVAD_FINAL_FACTOR,
		VVAD_FINAL_HIGH_FACTOR,
		VVAD_MIN_HIGH_THRESH,
		VVAD_AAM_MIN,
		VVAD_AAM_MAX} VVAD_PARAM;

/*
	We need to manage available channel id's on client side,
	verbio api can not handle this operations.
	
	Verbio API:
		- Grammars are referenced by id (given on vox_prevcbxx)
		- We need to specify a port (uniq dev id).
*/	

typedef struct{
	/* grammar name */
	const char *name;
	/* grammar path*/
	const char *path;
	/* grammar mode (ISOLATED / ABNF)*/ /*TODO: Allow CONNECTED?*/
	int mode;
}gram_info;

struct verbio_speech_info{
	/* verbosity */
	int verbose;
	/* default ASR config*/
	const char *conf;
	/* default ASR lang*/
	const char *lang;
	/* verbio device */
	int device;
	/* verbio rec device*/
	int recdev;
	/* client-side vad device */
	VAD_PARAM *vsddev;
	/* dafault grammar path*/
	const char *gram_path;
	/* gram_info. will allow us to directly reference
	  (using verbio's gram id) a grammar*/
	gram_info grammar[100];/* TODO: dynamic num of grammars */
	/* absolute timeout */
	int abs_timeout;
	/* sample count*/
	unsigned long int sample_count;
	/* sample rate x timeout*/
	unsigned long int fs_x_tout;
	/* record what user says*/
	int keep_rec_files;
	/* record vad detection*/
	int mark_rec_files;
	/* rec file */
	const char *rec_file_path;
	/* rec file */
	const char *rec_file_full_path;
	/* stop recognition on dtmf detection*/
	int stop_on_dtmf;
	/*************************/
	/****** VAD params *******/
	/*************************/
	int voice_detected;/*voice detection flag*/
	float init_sil;
	float max_sil;
	float max_ref;
	float min_ref;
	
	/**********************************/
	/****** Advanced VAD params *******/
	/**********************************/
	float low_factor;
	float high_factor;
	float final_factor;
	float final_high_factor;
	float min_high_thresh;
	float aa_min;
	float aa_max;
};

struct ast_speech_engine verbio_engine;

/* ********************************* */
/* ************ Helpers ************ */
/* ********************************* */
static int init_verbio_res_speech( void);
static void close_verbio_res_speech( void);

/*! \brief Helper function. Read config file*/
static struct ast_config* verbio_load_asterisk_config( void)
{
	#if ASTERISK_VERSION_NUM < AST_6 
	return ast_config_load( VERBIO_CFG); 
	#else
	struct ast_flags config_flags = { CONFIG_FLAG_WITHCOMMENTS };
	return ast_config_load( VERBIO_CFG, config_flags);
	#endif 
}

/*! \brief Helper function called on connection lost */
static void res_lost_conn_callback( const char *server)
{
	int i = 1;
	ast_log( LOG_ERROR, "Connection whith voxserver (%s) lost! Trying to reconnect...\n", server);
	while( 1)
	{
		close_verbio_res_speech();
		if( init_verbio_res_speech() <= 0)
		{
			/*ast_log( LOG_NOTICE, "it does not connect :(\n");*/
		}
		else
		{ 
			close_verbio_res_speech();
			if( init_verbio_res_speech() > 0)
			{
				ast_log( LOG_NOTICE, "connection with voxserver reestablished.\n");
				break;
			}
		}
		if( i < 120)
			sleep( 30 * i); //TODO: to config file
		else
			sleep( 3600);

		++i;
	}
}

/*! \brief Helper function. Init connection to voxserver  */
static int init_verbio_res_speech( void)
{
	struct ast_config 	*vcfg;
	const char 		*vcfg_primary_host = NULL;
	const char 		*vcfg_backup_host = NULL;
	const char 		*vcfg_default_asr_config;
	const char 		*vcfg_default_asr_lang;
	const char 		*vcfg_grammar_path = NULL;
	char 			gram_cache_path[MAXPATHLEN];
	unsigned int		done = 0;
	const unsigned int	retries = 5;/*TODO: to config file*/

	vcfg = verbio_load_asterisk_config();

	ast_log( LOG_NOTICE, "init verbio res_speech\n");

	if( !vcfg) 
	{
		ast_log(LOG_ERROR, "Error opening configuration file %s\n", 
								VERBIO_CFG);
		return -1;
	}

	/* If we perform a 'restart now' from CLI unload_module won't get called.
	   We need to disconnect prior to reconnect to verbio voxserver.
	*/
	usleep( 300000);/*be careful with (fast) reconnections*/

	if( !(vcfg_grammar_path = ast_variable_retrieve(vcfg, "asr", "grammar_path"))) 
	{
		ast_log( LOG_ERROR, "Could not read grammar_path option in %s. Please configure it.\n", VERBIO_CFG);
		ast_config_destroy( vcfg);
		return -1;
	}

	strcpy( gram_cache_path, vcfg_grammar_path);
	strcat( gram_cache_path, VERBIO_GRM_CACHE_DIR);
	if(mkdir( gram_cache_path, VERBIO_DIR_MODE) && errno != EEXIST) 
	{
		ast_log(LOG_WARNING, "mkdir '%s' failed: %s\n", gram_cache_path, strerror(errno));
		ast_config_destroy( vcfg);
		return -1;
	}

	/*TODO: also read secundary host */
	
	/* Set primary vox server ip*/
	if( !( vcfg_primary_host = ast_variable_retrieve(vcfg, "general", "primary_vox_server"))) 
		vcfg_primary_host = VERBIO_DEF_HOST;

	/* Set secondary vox server ip*/
	if( !( vcfg_backup_host = ast_variable_retrieve(vcfg, "general", "backup_vox_server"))) 
		vcfg_backup_host = VERBIO_DEF_HOST;

	if( !( vcfg_default_asr_config = ast_variable_retrieve(vcfg, "asr", "default_config"))) 
	{
		ast_log( LOG_ERROR, "Error: ASR 'default_config' option missing\n");
		ast_config_destroy( vcfg);
		return -1;
	}

	if( !( vcfg_default_asr_lang = ast_variable_retrieve(vcfg, "asr", "default_language"))) 
	{
		ast_log( LOG_ERROR, "Error: ASR 'default_language' option missing\n");
		ast_config_destroy( vcfg);
		return -1;
	}

	/* Init ASR */
	done = 0;
	unsigned int count = 0;
	while( (count < retries) && (done == 0))
	{
		vox_setparm( -1, VXGB_DEFSERVER, vcfg_primary_host);

		if( vox_asr_init( vcfg_default_asr_config, vcfg_default_asr_lang) != -1)
		{
			ast_log(LOG_NOTICE, "Success connecting to '%s' (conf: %s, lang: %s)\n",
										vcfg_primary_host, 
										vcfg_default_asr_config,
										vcfg_default_asr_lang);
			done = 1;
		}
		else	
		{
			sleep( 1);
			vox_setparm( -1, VXGB_DEFSERVER, vcfg_backup_host);
			if( vox_asr_init( vcfg_default_asr_config, vcfg_default_asr_lang) != -1) 
			{
				ast_log(LOG_NOTICE, "Success connecting to '%s' (conf: %s, lang: %s)\n",
										vcfg_backup_host, 
										vcfg_default_asr_config,
										vcfg_default_asr_lang);
				done = 1;
			}
			else
				sleep( 1);
		}
		++count;
	}	

	if( done == 1)
	{
		/*
		if( vox_getparm( -1, VXGB_DEFSERVER, (void *)&current_host) != -1)
			ast_log(LOG_NOTICE, "Success connecting to '%s' (conf: %s, lang: %s)\n",
										current_host, 
										vcfg_default_asr_config,
										vcfg_default_asr_lang);
		else
			ast_log(LOG_WARNING, "Could not get current voxserver's address.\n");
		*/
		vox_regsrvclose( res_lost_conn_callback);
		ast_config_destroy( vcfg);
		return 1;
	}
	else 
	{
		ast_log(LOG_ERROR, "Could not connect to Verbio voxserver %s and %s [%s]\n",
							vcfg_primary_host, 
							vcfg_backup_host, 
							ATVOX_ERRMSGP( -1));
		ast_config_destroy( vcfg);
		return -1;/*do not stop asterisk startup*/
	}
}

/*! \brief Helper function. Close connection to voxserver  */
static void close_verbio_res_speech( void)
{
	ast_log(LOG_NOTICE, "Closing connection to Verbio voxserver..\n");
	vox_libclose();
}

/*! \brief Helper function. Copy file*/
static int verbio_copy_file(char *infile, char *outfile)
{
	int ifd;
	int ofd;
	int res;
	int len;
	char buf[4096];

	if(( ifd = open( infile, O_RDONLY)) < 0) 
	{
		/*ast_log( LOG_WARNING, "Unable to open %s in read-only mode.\n", infile);*/
		ast_log(LOG_WARNING, "Unable to open %s in read-only mode: %s\n", infile, strerror(errno));
		return -1;
	}
	if(( ofd = open( outfile, O_WRONLY | O_TRUNC | O_CREAT, VERBIO_FILE_MODE)) < 0) 
	{
		/*ast_log(LOG_WARNING, "Unable to open %s in write-only mode.\n", outfile);*/
		ast_log(LOG_WARNING, "Unable to open %s in write-only mode: %s\n", outfile, strerror(errno));
		close(ifd);
		return -1;
	}
	do 
	{
		len = read(ifd, buf, sizeof(buf));
		if (len < 0) 
		{
			/*ast_log( LOG_WARNING, "Read failed on %s.\n", infile);*/
			ast_log(LOG_WARNING, "Read failed on %s: %s\n", infile, strerror(errno));
			close( ifd);
			close( ofd);
			unlink( outfile);
		}
		if (len) 
		{
			res = write( ofd, buf, len);
			if( errno == ENOMEM || errno == ENOSPC || res != len) 
			{
				/*ast_log( LOG_WARNING, "Write failed on %s (%d of %d).\n", outfile, res, len);*/
				ast_log(LOG_WARNING, "Write failed on %s (%d of %d): %s\n", outfile, res, len, strerror(errno));
				close( ifd);
				close( ofd);
				unlink( outfile);
			}
		}
	} while( len);
	close( ifd);
	close( ofd);
	return 0;
}

/*! \brief Helper function. Set verbosity flag*/
static void verbio_set_verbose( struct ast_speech *speech, int v)
{
	struct verbio_speech_info *verbio_info;
	verbio_info = (struct verbio_speech_info *)speech->data;
	if( verbio_info != NULL)
		verbio_info->verbose = v;
	else
		ast_log( LOG_ERROR, "Could not modify verbosity\n");
}

/*! \brief Helper function. Get verbosity flag*/
static int verbio_get_verbose( struct ast_speech *speech)
{
	struct verbio_speech_info *verbio_info;
	verbio_info = (struct verbio_speech_info *)speech->data;
	if( verbio_info != NULL)
		return verbio_info->verbose;
	else
		return -1;
}

/*! \brief Helper function. Get current Verbio device (previously set)*/
static int verbio_get_curr_dev( struct ast_speech *speech)
{
	int dev = -2; /* -1 reserved to reference all channels*/
	struct verbio_speech_info *verbio_info;
	verbio_info = (struct verbio_speech_info *)speech->data;

	if( verbio_info != NULL)
		dev = verbio_info->device;
	else
		ast_log( LOG_ERROR, "Could not get a valid Verbio device\n");

	return dev;
}

/*! \brief Helper function. Set current Verbio recdevice*/
static void verbio_set_recdev( struct ast_speech *speech, int rec)
{
	struct verbio_speech_info *verbio_info;
	verbio_info = (struct verbio_speech_info *)speech->data;
	if( verbio_info != NULL)
		verbio_info->recdev = rec;
	else
		ast_log( LOG_ERROR, "Could not set a valid Verbio rec device\n");
}

/*! \brief Helper function. Get current Verbio recdevice*/
static int verbio_get_recdev( struct ast_speech *speech)
{
	int rd = 0;
	struct verbio_speech_info *verbio_info;
	verbio_info = (struct verbio_speech_info *)speech->data;
	
	if( verbio_info != NULL)
		rd = verbio_info->recdev;
	else
		ast_log( LOG_ERROR, "Could not get a valid Verbio rec device\n");

	return rd;
}

/*! \brief Helper function. Set current Verbio vsddevice (client-side VAD)*/
static void verbio_set_vsddev( struct ast_speech *speech, VAD_PARAM *v)
{
	struct verbio_speech_info *verbio_info;
	verbio_info = (struct verbio_speech_info *)speech->data;

	if( verbio_info != NULL)
		verbio_info->vsddev = v;
	else
		ast_log( LOG_ERROR, "Could not set a valid Verbio vsd device\n");
}

/*! \brief Helper function. Get current Verbio vsddevice (client-side VAD)*/
static VAD_PARAM* verbio_get_vsddev( struct ast_speech *speech)
{
	VAD_PARAM *rd = NULL;
	struct verbio_speech_info *verbio_info;
	verbio_info = (struct verbio_speech_info *)speech->data;

	if( verbio_info != NULL)
		rd = verbio_info->vsddev;
	else
		ast_log( LOG_ERROR, "Could not get a valid Verbio vsd device\n");

	return rd;
}

/*! \brief Helper function. Set ASR configuration*/
static int verbio_set_curr_config( struct ast_speech *speech, const char *config)
{
	struct verbio_speech_info *verbio_info;
	verbio_info = (struct verbio_speech_info *)speech->data;

	if( verbio_info != NULL)
	{
		verbio_info->conf = strdup(config);
		return 0;
	}
	else
	{
		ast_log( LOG_ERROR, "Could not set ASR configuration to %s\n", config);
		return -1;
	}
}

/*! \brief Helper function. Get ASR configuration*/
static const char* verbio_get_curr_config( struct ast_speech *speech)
{
	struct verbio_speech_info *verbio_info;
	verbio_info = (struct verbio_speech_info *)speech->data;

	if( verbio_info != NULL)
		return verbio_info->conf;
	else
	{
		ast_log( LOG_ERROR, "Could not get ASR configuration\n"); 
		return NULL;
	}
}

/*! \brief Helper function. Set ASR language*/
static int verbio_set_curr_lang( struct ast_speech *speech, const char *lang)
{
	struct verbio_speech_info *verbio_info;
	verbio_info = (struct verbio_speech_info *)speech->data;

	if( verbio_info != NULL)
	{
		verbio_info->lang = strdup(lang);
		return 0;
	}
	else
	{
		ast_log( LOG_ERROR, "Could not set ASR language to %s\n", lang);
		return -1;
	}
}

/*! \brief Helper function. Get ASR language*/
static const char* verbio_get_curr_lang( struct ast_speech *speech)
{
	struct verbio_speech_info *verbio_info;
	verbio_info = (struct verbio_speech_info *)speech->data;

	if( verbio_info != NULL)
		return verbio_info->lang;
	else
	{
		ast_log( LOG_ERROR, "Could not get ASR language\n");
		return NULL;
	}
}

/*! \brief Helper function. Set ASR language and config from iso lang code*/
static int verbio_set_curr_lang_conf( struct ast_speech *speech, const char *isolang)
{
	struct verbio_speech_info *verbio_info;
	verbio_info = (struct verbio_speech_info *)speech->data;

	if( verbio_info != NULL)
	{
		char clang[40];
		char cconf[40];
		/*char avconfs[200];*/

		/*match iso lang*/
		if( !strcasecmp( isolang, "eu-ES"))
			strcpy( clang, "eu");
                else if( !strcasecmp( isolang, "ca-ES"))
			strcpy( clang, "ca");	
		else if( !strcasecmp( isolang, "en-US"))
			strcpy( clang, "en-us");	
		else if( !strcasecmp( isolang, "es-AR"))
			strcpy( clang, "es-ar");	
		/*else if( !strcasecmp( isolang, "es-BO"))
			strcpy( clang, "");*/	
		else if( !strcasecmp( isolang, "es-CL"))
			strcpy( clang, "es-cl");	
		else if( !strcasecmp( isolang, "es-CO"))
			strcpy( clang, "es-co");	
		/*else if( !strcasecmp( isolang, "es-CR"))
			strcpy( clang, "");	
		else if( !strcasecmp( isolang, "es-DO"))
			strcpy( clang, "");	
		else if( !strcasecmp( isolang, "es-EC"))
			strcpy( clang, "");*/	
		else if( !strcasecmp( isolang, "es-ES"))
			strcpy( clang, "es");	
		/*else if( !strcasecmp( isolang, "es-GT"))
			strcpy( clang, "");	
		else if( !strcasecmp( isolang, "es-HN"))
			strcpy( clang, "");*/	
		else if( !strcasecmp( isolang, "es-MX"))
			strcpy( clang, "es-mx");	
		/*else if( !strcasecmp( isolang, "es-NI"))
			strcpy( clang, "");	
		else if( !strcasecmp( isolang, "es-PA"))
			strcpy( clang, "es-co");	
		else if( !strcasecmp( isolang, "es-PE"))
			strcpy( clang, "");	
		else if( !strcasecmp( isolang, "es-PR"))
			strcpy( clang, "es-ve");	
		else if( !strcasecmp( isolang, "es-PY"))
			strcpy( clang, "es-ar");	
		else if( !strcasecmp( isolang, "es-SV"))
			strcpy( clang, "");	
		else if( !strcasecmp( isolang, "es-UY"))
			strcpy( clang, "es-ar");*/	
		else if( !strcasecmp( isolang, "es-VE"))
			strcpy( clang, "es-ve");	
		else if( !strcasecmp( isolang, "fr-FR"))
			strcpy( clang, "fr");	
		else if( !strcasecmp( isolang, "gl-ES"))
			strcpy( clang, "ga");	
		else if( !strcasecmp( isolang, "pt-BR"))
			strcpy( clang, "pt-br");	
		else if( !strcasecmp( isolang, "pt-PT"))
			strcpy( clang, "pt");	
		else
		{
			ast_log( LOG_WARNING, "Language %s not supported\n", isolang);
			ast_log( LOG_NOTICE, "Available languages:es-ES,eu-ES,ca-ES,gl-ES,pt-BR,pt-PT,en-US,fr-FR,es-MX,es-AR,es-CL,es-CO,es-VE\n");
			return -1;
		}


		/* Get config from language */
		/* 1 - Get available configs from server */
		char *asrconf;
		if( vox_getparm(-1, VXGB_START_CONF, &asrconf) != 0)
		{
			ast_log( LOG_WARNING, "Could not get available configs from server.\n");
			return -1;	
		}

		if( verbio_get_verbose( speech) > 0)
		{
			ast_log( LOG_NOTICE, "Available configs: %s\n", asrconf);
			ast_log( LOG_NOTICE, "Current language : %s\n", clang);
		}

		/* 2 - Search config for current language */
		if( strstr( asrconf, clang))
		{
			if( !strcmp( clang, "es"))
			{
				/*search for es, es_ca, es_eu, es_ga, es_ca_eu_ga*/
				if( strstr( asrconf, "es_ca_eu_ga,"))
					strcpy( cconf, "es_ca_eu_ga");
				else if( strstr( asrconf, "es_ca,"))
					strcpy( cconf, "es_ca");
				else if( strstr( asrconf, "es_eu,"))
					strcpy( cconf, "es_eu");
				else if( strstr( asrconf, "es_ga,"))
					strcpy( cconf, "es_ga");
				else if( strstr( asrconf, "es,"))
					strcpy( cconf, "es");
				else if( verbio_get_verbose( speech) > 0) 
					ast_log( LOG_NOTICE, "Nothing for lang \"es\" found %s", asrconf);
			}
			else if( !strcmp( clang, "ca"))
			{
				/*search for es_ca, es_ca_eu_ga*/
				if( strstr( asrconf, "es_ca_eu_ga,"))
					strcpy( cconf, "es_ca_eu_ga");
				else if( strstr( asrconf, "es_ca,"))
					strcpy( cconf, "es_ca");
				else if( verbio_get_verbose( speech) > 0) 
					ast_log( LOG_NOTICE, "Nothing for lang \"ca\" found %s", asrconf);
			}
			else if( !strcmp( clang, "eu"))
			{
				/*search for es_eu, es_ca_eu_ga*/
				if( strstr( asrconf, "es_ca_eu_ga,"))
					strcpy( cconf, "es_ca_eu_ga");
				else if( strstr( asrconf, "es_eu,"))
					strcpy( cconf, "es_eu");
				else if( verbio_get_verbose( speech) > 0) 
					ast_log( LOG_NOTICE, "Nothing for lang \"eu\" found %s", asrconf);
			}
			else if( !strcmp( clang, "ga"))
			{
				/*search for es_ga, es_ca_eu_ga*/
				if( strstr( asrconf, "es_ca_eu_ga,"))
					strcpy( cconf, "es_ca_eu_ga");
				else if( strstr( asrconf, "es_ga,"))
					strcpy( cconf, "es_ga");
				else if( verbio_get_verbose( speech) > 0) 
					ast_log( LOG_NOTICE, "Nothing for lang \"ga\" found %s", asrconf);
			}
			else
			{
				if( verbio_get_verbose( speech) > 0) 
					ast_log( LOG_NOTICE, "lang = conf\n");	
				char tmplang[41];
				strcpy( tmplang, clang);
				strcat( tmplang, ",");
				if( strstr( asrconf, tmplang))
					strcpy( cconf, clang);
				else
				{
					ast_log( LOG_ERROR, "No suittable config found for language \"%s\"\n", clang);	
					return -1;
				}
			}
 		}
 		else
		{
			ast_log( LOG_ERROR, "Language %s not found in configs %s\n", isolang, asrconf);
			return -1;
                }
		
		if( verbio_get_verbose( speech) > 0) 
			ast_log( LOG_NOTICE, "Language:\"%s\" config:\"%s\"\n", clang, cconf);

		verbio_info->lang = strdup(clang);
		verbio_info->conf = strdup(cconf);
		return 0;
	}
	else
	{
		ast_log( LOG_ERROR, "Could not set ASR language to %s\n", isolang);
		return -1;
	}
}


#if 0
static int verbio_set_global_grammar( struct ast_speech *speech, const char *grm)
{
	/*TODO*/
}
#endif

/*! \brief Helper function. Set default grammar path*/
static int verbio_set_def_gram_path( struct ast_speech *speech, const char *p)
{
	struct verbio_speech_info *verbio_info;
	verbio_info = (struct verbio_speech_info *)speech->data;

	if( verbio_info != NULL)
	{
		verbio_info->gram_path = strdup( p);
		return 0;
	}
	else
	{
		ast_log( LOG_ERROR, "Could not set default grammar path to %s\n", p);
		return -1;
	}
}

/*! \brief Helper function. Get default grammar path*/
static const char* verbio_get_def_gram_path( struct ast_speech *speech)
{
	struct verbio_speech_info *verbio_info;
	verbio_info = (struct verbio_speech_info *)speech->data;

	if( verbio_info != NULL)
		return verbio_info->gram_path;
	else
	{
		ast_log( LOG_ERROR, "Could not get default grammar path\n");
		return NULL;
	}
}

/*! \brief Helper function. Set VAD (Voice Activity Detector) param*/
static int verbio_set_curr_vad_param( struct ast_speech *speech, VVAD_PARAM p, float v)
{
	struct verbio_speech_info *verbio_info;
	verbio_info = (struct verbio_speech_info *)speech->data;

	if( !verbio_info)
	{
		ast_log( LOG_ERROR, "Error setting VAD param %d\n", p);
		return -1;
	}

	if( v < 0)
	{
		ast_log( LOG_ERROR, "Not setting negative VAD param %f\n", v);
		return -1;
	}

	switch( p) {
		case VVAD_INIT_SIL:  
			verbio_info->init_sil = v;
		break;
		case VVAD_MAX_SIL:  
			verbio_info->max_sil = v;
		break;
		case VVAD_MAX_REF:  
			verbio_info->max_ref = v;
		break;
  		case VVAD_MIN_REF:  
			verbio_info->min_ref = v;
		break;
  		case VVAD_LOW_FACTOR:  
			verbio_info->low_factor = v;
		break;
  		case VVAD_HIGH_FACTOR:  
			verbio_info->high_factor = v;
		break;
  		case VVAD_FINAL_FACTOR:  
			verbio_info->final_factor = v;
		break;
  		case VVAD_FINAL_HIGH_FACTOR:  
			verbio_info->final_high_factor = v;
		break;
  		case VVAD_MIN_HIGH_THRESH:  
			verbio_info->min_high_thresh = v;
		break;
  		case VVAD_AAM_MIN:  
			verbio_info->aa_min = v;
		break;
  		case VVAD_AAM_MAX:  
			verbio_info->aa_max = v;
		break;
  	}

	return 0;
}

/*! \brief Helper function. Set VAD (Voice Activity Detector) mode */
static int verbio_set_vad_mode( struct ast_speech *speech, const char *mode)
{
	struct verbio_speech_info *verbio_info;
	verbio_info = (struct verbio_speech_info *)speech->data;

	if( !verbio_info)
	{
		ast_log( LOG_ERROR, "Error setting VAD mode to %s\n", mode);
		return -1;
	}

	if( verbio_get_verbose( speech) > 0)
		ast_log( LOG_NOTICE, "Setting VAD mode to %s\n", mode);

	float mfactor = 1.0;

	if( !strcmp( mode, "quiet"))
		 mfactor = 1.2;
	else if( !strcmp( mode, "standard"))
        	mfactor = VSTD_FACTOR;
	else if( !strcmp( mode, "noisy"))
        	mfactor = VN_FACTOR;
	else if( !strcmp( mode, "hysteria"))
        	mfactor = VH_FACTOR;
	else
	{
 		ast_log( LOG_ERROR, "Unknown VAD mode %s. Defaulting to standard mode.\n", mode);
		mfactor = VSTD_FACTOR; 
		/*return -1;*/
	}

	verbio_info->low_factor 	= VERBIO_LOW_FACTOR * mfactor;
	verbio_info->high_factor 	= VERBIO_HIGH_FACTOR * mfactor;
	verbio_info->final_factor 	= VERBIO_FINAL_FACTOR * mfactor;
	verbio_info->final_high_factor 	= VERBIO_FINAL_HIGH_FACTOR * mfactor;
	verbio_info->min_high_thresh 	= VERBIO_MIN_HIGH_THRESH * mfactor * 0.5;
	verbio_info->aa_min 		= VERBIO_AAM_MIN * mfactor;
	verbio_info->aa_max 		= VERBIO_AAM_MAX * mfactor;

	/*Also modify server-side VAD params (env. noise will also affect final silence detection)*/
	verbio_info->min_ref		= VERBIO_MIN_REF * mfactor * 0.5;
	verbio_info->max_ref		= VERBIO_MAX_REF;

	return 0;
}

/*! \brief Helper function. Get VAD (Voice Activity Detector) param*/
static float verbio_get_curr_vad_param( struct ast_speech *speech, VVAD_PARAM p)
{
	struct verbio_speech_info *verbio_info;
	verbio_info = (struct verbio_speech_info *)speech->data;

	if( !verbio_info)
	{
		ast_log( LOG_ERROR, "Error getting VAD param %d\n", p);
		return -1;
	}

	switch( p) {
		case VVAD_INIT_SIL:  
			return verbio_info->init_sil;
		break;
		case VVAD_MAX_SIL:  
			return verbio_info->max_sil;
		break;
		case VVAD_MAX_REF:  
			return verbio_info->max_ref;
		break;
  		case VVAD_MIN_REF:  
			return verbio_info->min_ref;
		break;
		case VVAD_LOW_FACTOR:  
			return verbio_info->low_factor;
		break;
  		case VVAD_HIGH_FACTOR:  
			return verbio_info->high_factor;
		break;
  		case VVAD_FINAL_FACTOR:  
			return verbio_info->final_factor;
		break;
  		case VVAD_FINAL_HIGH_FACTOR:  
			return verbio_info->final_high_factor;
		break;
  		case VVAD_MIN_HIGH_THRESH:  
			return verbio_info->min_high_thresh;
		break;
  		case VVAD_AAM_MIN:  
			return verbio_info->aa_min;
		break;
  		case VVAD_AAM_MAX:  
			return verbio_info->aa_max;
		break;
  	}

	return -1;
}

/*! \brief Helper function. Check VAD status*/
static int verbio_get_voice_detected( struct ast_speech *speech)
{
	struct verbio_speech_info *verbio_info;
	verbio_info = (struct verbio_speech_info *)speech->data;

	if( verbio_info != NULL)
		return verbio_info->voice_detected;
	else
	{
		ast_log( LOG_ERROR, "Error checking voice detection flag\n");
		return -1;
	}
}

/*! \brief Helper function. Modify VAD status*/
static int verbio_set_voice_detected( struct ast_speech *speech, int v)
{
	struct verbio_speech_info *verbio_info;
	verbio_info = (struct verbio_speech_info *)speech->data;

	if( verbio_info != NULL)
	{
		verbio_info->voice_detected = v;
		return 0;
	}
	else
	{
		ast_log( LOG_ERROR, "Error setting voice detection\n");
		return -1;
	}
}

/*! \brief Helper function. Add grammar to speech engine grammar list.*/
static int verbio_add_grammar( struct ast_speech *speech, 
					const char *grammar_name, 
					const char *grammar_path, 
					int grammar_id)
{
	int verbose = verbio_get_verbose( speech);
	struct verbio_speech_info *verbio_info;
	verbio_info = (struct verbio_speech_info *)speech->data;

	if( grammar_id > (VERBIO_MAX_GRAM_NUM -1))
	{
		ast_log( LOG_ERROR, "Error: Max grammar number reached [%d]\n", grammar_id);
		return -1;
	}
	
	if( verbio_info != NULL)
	{	
		verbio_info->grammar[grammar_id].name = strdup(grammar_name);
		verbio_info->grammar[grammar_id].path = strdup(grammar_path);
		
		if( verbose > 0)
		{
			ast_log( LOG_NOTICE, "Added grammar: %s %s id:%d\n", 
								grammar_name,  
								grammar_path,
								grammar_id);
		}
		return 0;
	}
	else
	{
		ast_log( LOG_ERROR, "Could not add grammar to list\n");
		return -1;
	}
	
}

/*! \brief Helper function. Get previously loaded grammar id (needed to reference grammars).*/
static int verbio_get_grammar_id( struct ast_speech *speech, char *name/*, char* path*/)
{
	int verbose = verbio_get_verbose( speech);
	int gid = -2; /* -1 may be reserved (to unload all grammars)*/
	struct verbio_speech_info *verbio_info;
	verbio_info = (struct verbio_speech_info *)speech->data;
	int done = 0;
	int id = 0;
	while( (!done) && (id < VERBIO_MAX_GRAM_NUM))/*TODO: dyn number of grammars?*/
	{
		if( verbio_info->grammar[id].name == NULL)
			break;
		
		if( !strcmp( name, verbio_info->grammar[id].name))
		{
			gid = id;
			done = 1;
			if( verbose > 0)
				ast_log( LOG_NOTICE, "Found grammar %s id: %d\n", name, id);
		}
		++id;
	}

	return gid;
}

/*! \brief Helper function. Get grammar mode (abnf, text).*/
static int verbio_get_grammar_mode( struct ast_speech *speech, const char *grammar_path)
{
	int verbose = verbio_get_verbose( speech);
	int mode = GVX_ABNF;

	/* auto grammar mode (abnf, txt) detection*/
	int last_indx = strlen(grammar_path);

	if( verbose > 0)
		ast_log( LOG_NOTICE, "Getting grammar mode for: %s\n", grammar_path);

	/* Isolated (word list) grammars must have 'txt' extension*/
	/* TODO: more 'elegant' detection.*/
	if( strstr( grammar_path, "builtin:grammar/"))
	{
		if( verbose > 0)
			ast_log( LOG_NOTICE, "ABNF/builtin mode\n");
	}
	else if( toupper( grammar_path[last_indx - 1]) == 'T' && 
			toupper( grammar_path[last_indx - 2]) == 'X' && 
			toupper( grammar_path[last_indx - 3]) == 'T' &&
				 grammar_path[last_indx - 4] == '.')
	{
		if( verbose > 0)
			ast_log( LOG_NOTICE, "ISOLATED mode\n");
		mode = GVX_ISOLATED;
	}
	else
	{
		if( verbose > 0)
			ast_log( LOG_NOTICE, "ABNF mode\n");
	}

	return mode;
}

/*! \brief Helper function. Remove grammar from list*/
static void verbio_remove_gram( struct ast_speech *speech, int g)
{
	int verbose = verbio_get_verbose( speech);
	struct verbio_speech_info *verbio_info;
	verbio_info = (struct verbio_speech_info *)speech->data;
	
	if( verbio_info != NULL)
	{	
		if( verbio_info->grammar[g].name != NULL)
			verbio_info->grammar[g].name = "";

		if( verbio_info->grammar[g].path != NULL)
			verbio_info->grammar[g].path = "";
		
		if( verbose > 0)
			ast_log( LOG_NOTICE, "Grammar id :%d removed\n", g);
	}
	else
		ast_log( LOG_ERROR, "Could not remove grammar id %d\n", g);
}

/*! \brief Helper function. Increase sample count (processed samples)*/
static void verbio_incr_sample_count( struct ast_speech *speech, int len)
{
	/* If len = -1 --> set sample counter to 0 */
	struct verbio_speech_info *verbio_info;
	verbio_info = (struct verbio_speech_info *)speech->data;

	if( verbio_info != NULL)
	{
		if( len < 0)
			verbio_info->sample_count = 0;
		else
			verbio_info->sample_count += len;
		/*ast_log( LOG_NOTICE, "sample count set to %ld\n", 
					verbio_info->sample_count);*/
	}
	else
		ast_log( LOG_ERROR, "Could not increase sample count\n");
}

/*! \brief Helper function. Get number of processed samples*/
static unsigned long int verbio_get_sample_count( struct ast_speech *speech)
{
	int retval = -1;
	struct verbio_speech_info *verbio_info;
	verbio_info = (struct verbio_speech_info *)speech->data;

	if( verbio_info != NULL)
		retval = verbio_info->sample_count;
	else
		ast_log( LOG_ERROR, "Could not get sample count\n");
	
	return retval;
}

/*! \brief Helper function. Set absolute timeout*/
static int verbio_set_abs_timeout( struct ast_speech *speech, int t)
{
	int verbose = verbio_get_verbose( speech);
	struct verbio_speech_info *verbio_info;
	verbio_info = (struct verbio_speech_info *)speech->data;

	if( verbio_info != NULL)
	{
		verbio_info->abs_timeout = t;
		verbio_info->fs_x_tout = t * 8000 * 2;
		
		if( verbose > 0)
			ast_log( LOG_NOTICE, "Abs timeout set to %d\n", 
						verbio_info->abs_timeout);
		return 0;
	}
	else
	{
		ast_log( LOG_ERROR, "Could not set absolute timeout\n");
		return -1;
	}
}

#if 0   /*Not needed*/
/*! \brief Helper function. Get absolute timeout*/
static int verbio_get_abs_timeout( struct ast_speech *speech)
{
	int retval = -1;
	struct verbio_speech_info *verbio_info;
	verbio_info = (struct verbio_speech_info *)speech->data;

	if( verbio_info != NULL)
		retval = verbio_info->abs_timeout;
	else
		ast_log( LOG_ERROR, "Could not get timeout\n");
	
	return retval;
}
#endif

/*! \brief Helper function. Get timeout in samples*/
static unsigned long int verbio_get_fs_x_tout( struct ast_speech *speech)
{
	int retval = -1;
	struct verbio_speech_info *verbio_info;
	verbio_info = (struct verbio_speech_info *)speech->data;

	if( verbio_info != NULL)
		retval = verbio_info->fs_x_tout;
	else
		ast_log( LOG_ERROR, "Could not get fs x timeout\n");
	
	return retval;
}

#if 0
/*! \brief Helper function. Set new result*/
static int verbio_add_result( struct ast_speech *speech, struct verbio_speech_info *verbio_info, int gid, struct ast_speech_result *new_result, const char *text, int score)
{
	/*struct ast_speech_result *new_result = NULL;*/

#if ASTERISK_VERSION_NUM < AST_6 
/*	result->next = ast_calloc(1, sizeof(*speech->results));
	new_result = result->next;*/
	new_result = ast_calloc(1, sizeof(*speech->results));
#else
	/*new_result = AST_LIST_NEXT(result, list);*/
	new_result = ast_calloc(1, sizeof(*speech->results));
#endif 

	new_result->text = strdup( text);
	new_result->score = score;

	ast_log(LOG_WARNING, "Result %s %d \n", new_result->text, new_result->score);

	/*result->grammar = strdup(gram);*/
	if( verbio_info->grammar[gid].name != NULL)
		new_result->grammar = strdup( verbio_info->grammar[gid].name);
	else
	{
		ast_log(LOG_ERROR, "Error: Grammar name is null for gid = %d \n", gid);
		new_result->grammar = "NULL";
	}	
	
	return 0;

}
#endif

/*! \brief Helper function. Sets recognition result (fill results struct)*/
static int verbio_set_asr_result( struct ast_speech *speech)
{	
	int 	verbose = verbio_get_verbose( speech);
	int	dev = 0;
	int 	nind = 0;
	int 	index[VERBIO_MAXINDEX+1];
	float 	score[VERBIO_MAXINDEX+1];
	int 	i 	= 0;
	struct 	ast_speech_result *result = NULL;
	
	speech->state = AST_SPEECH_STATE_WAIT;

	/* ----------------------------- */
	/* ----- Get verbio device ----- */
	/* ----------------------------- */
	dev = verbio_get_curr_dev( speech);
	if( dev == -2)
		return -1;
	
	/* -------------------------------------- */
	/* ----- Get result from recognizer ----- */
	/* -------------------------------------- */
	if(( nind = vox_recind( dev, VERBIO_MAXINDEX, index, score, 0)) < 0)
		ast_log( LOG_ERROR, "Error: vox_recind %s\n", ATVOX_ERRMSGP( dev));

	/* nind = number of recognized words */
	if( verbose > 0)
		ast_log( LOG_NOTICE, "Number of recognized words: %d\n", nind);
	
	if( nind == 0) 
	{
		/* No words recognized */
		/* TODO: can we distinguish between noword and noinput?*/
		/* TODO:speaker may also said an invalid (not in gram) word*/
		speech->flags = AST_SPEECH_SPOKE; 
	}
	else 
	{
		int gid = ATVOX_IVCB( dev);
		struct verbio_speech_info *verbio_info;
		/*char *gram = NULL;*/
		/*char gram[PATH_MAX];*/
	
		verbio_info = (struct verbio_speech_info *)speech->data;
		if( verbio_info == NULL)
		{
			ast_log(LOG_ERROR, "Error: Grammar id %d Not found  \n", gid);
			return 0;/*TODO: return -1 ??*/
		}
		/*gram = verbio_info->grammar[gid].name;*/
		/*strcpy( gram, verbio_info->grammar[gid].name);*/
		
		/* First result */
		speech->results = ast_calloc(1, sizeof(*speech->results));
		result = speech->results;
		result->text = strdup(vox_word(dev, index[0]));
		result->score = (int)score[0];
		/*result->grammar = strdup( gram);*/
		if( verbio_info->grammar[gid].name != NULL)
			result->grammar = strdup( verbio_info->grammar[gid].name);
		else
		{
			ast_log(LOG_ERROR, "Error: Grammar name is null for gid = %d \n", gid);
			result->grammar = "NULL";
			/*return 0;*//*TODO: return -1 ??*/
		}	

		/* for each recognized word...*/
		for( i = 1; i < nind; i++) 
		{

		#if ASTERISK_VERSION_NUM < AST_6 
			result->next = ast_calloc(1, sizeof(*speech->results));
			result = result->next;
		#else
			result->list.next = ast_calloc(1, sizeof(*speech->results));
			/*result = AST_LIST_NEXT(result, list);*/
			/*result = ast_calloc(1, sizeof(*speech->results));*/
			result = result->list.next;
		#endif 

			result->text = strdup(vox_word(dev, index[i]));
			result->score = (int)score[0];
			/*result->grammar = strdup(gram);*/
			if( verbio_info->grammar[gid].name != NULL)
				result->grammar = strdup( verbio_info->grammar[gid].name);
			else
			{
				ast_log(LOG_ERROR, "Error: Grammar name is null for gid = %d \n", gid);
				result->grammar = "NULL";
				/*return 0;*//*TODO: return -1 ??*/
			}
		}
		speech->flags = AST_SPEECH_HAVE_RESULTS;
	}
	speech->state = AST_SPEECH_STATE_DONE;

	/* reset sample counter */
	verbio_incr_sample_count(speech, -1);

	return 0;
}

/*! \brief Helper function. Modify keep rec files flag*/
static void verbio_set_keep_rec_files( struct ast_speech *speech, int k)
{
	int verbose = verbio_get_verbose( speech);
	struct verbio_speech_info *verbio_info;
	verbio_info = (struct verbio_speech_info *)speech->data;

	if( verbio_info != NULL)
	{
		verbio_info->keep_rec_files = k;
		if( verbose > 0)
			ast_log( LOG_NOTICE, "Keeping rec files\n");
	}
	else
	{
		ast_log( LOG_ERROR, "Could no set keep rec files\n");
	}
}

/*! \brief Helper function. Get keep rec files flag*/
static int verbio_get_keep_rec_files( struct ast_speech *speech)
{
	int retval = -1;
	struct verbio_speech_info *verbio_info;
	verbio_info = (struct verbio_speech_info *)speech->data;

	if( verbio_info != NULL)
		retval = verbio_info->keep_rec_files;
	else
		ast_log( LOG_ERROR, "Could not get keep rec files\n");
	
	return retval;
}

/*! \brief Helper function. Modify mark rec files flag*/
static void verbio_set_mark_rec_files( struct ast_speech *speech, int k)
{
	int verbose = verbio_get_verbose( speech);
	struct verbio_speech_info *verbio_info;
	verbio_info = ( struct verbio_speech_info *)speech->data;

	if( verbio_info != NULL)
	{
		verbio_info->mark_rec_files = k;
		if( verbose > 0)
			ast_log( LOG_NOTICE, "Marking rec files\n");
	}
	else
	{
		ast_log( LOG_ERROR, "Could no set mark rec files\n");
	}
}

/*! \brief Helper function. Get mark rec files flag*/
static int verbio_get_mark_rec_files( struct ast_speech *speech)
{
	int retval = -1;
	struct verbio_speech_info *verbio_info;
	verbio_info = ( struct verbio_speech_info *)speech->data;

	if( verbio_info != NULL)
		retval = verbio_info->mark_rec_files;
	else
		ast_log( LOG_ERROR, "Could not get mark rec files\n");
	
	return retval;
}

/*! \brief Helper function. Set full path (of recorded file)*/
static int verbio_set_rec_file( struct ast_speech *speech)
{
	int verbose = verbio_get_verbose( speech);
	struct verbio_speech_info *verbio_info;
	verbio_info = (struct verbio_speech_info *)speech->data;
	struct ast_config 	*vcfg = NULL;	
	const char		*vcfg_recorded_files_exten = NULL;
	const char		*vcfg_recorded_files_path = NULL;
	
        vcfg = verbio_load_asterisk_config();
	
	if( !vcfg) 
	{
		ast_log(LOG_ERROR, "Error opening configuration file %s\n", VERBIO_CFG);
		return -1;
	}

	vcfg_recorded_files_exten = ast_variable_retrieve(vcfg, "debug", "recorded_files_exten");
	vcfg_recorded_files_path = ast_variable_retrieve(vcfg, "debug", "recorded_files_path");

	if( verbio_info != NULL)
	{
		char tmp[20];
		char filename[30];
		char path[PATH_MAX];
		time_t t = time (NULL);
		int dev = verbio_get_curr_dev( speech);
		strcpy( filename, base_fname);
		if( snprintf( tmp, 20, "%d", dev) > 20) 
			ast_log( LOG_WARNING, "Filename may be truncated\n"); 
		strcat( filename, tmp);
		strcat( filename, "-");
		if( snprintf( tmp, 20, "%d", (int)t) > 20) 
			ast_log( LOG_WARNING, "Filename may be truncated\n"); 
		strcat( filename, tmp);
		strcat( filename, ".");
		if( vcfg_recorded_files_exten)
			strcat( filename, vcfg_recorded_files_exten);
		else
			strcat( filename, exten);
				
		/*strcpy( path, verbio_info->rec_file_path);*/
		strcpy( path, vcfg_recorded_files_path);
		strcat( path, "/");
		strcat( path, filename);
                
		verbio_info->rec_file_full_path = strdup( path);
		if( verbose > 0)
			ast_log( LOG_NOTICE, "Rec file set to %s\n", path);
	}
	else
	{
		ast_log( LOG_ERROR, "Could no set keep rec files\n");
		ast_config_destroy( vcfg);
		return -1;
	}

	ast_config_destroy( vcfg);
	return 0;
}

/*! \brief Helper function. Stop recognition on dtmf detection*/
static void verbio_set_stop_on_dtmf( struct ast_speech *speech, int s)
{
	int verbose = verbio_get_verbose( speech);
	struct verbio_speech_info *verbio_info;
	verbio_info = (struct verbio_speech_info *)speech->data;

	if( verbio_info != NULL)
	{
		verbio_info->stop_on_dtmf = s;
		if( verbose > 0)
			ast_log( LOG_NOTICE, "Stop recognition on dtmf activated.\n");
	}
	else
	{
		ast_log( LOG_ERROR, "Could not activate stop on dtmf.\n");
	}
}

#if __BYTE_ORDER == __LITTLE_ENDIAN
#define htoll(b) (b)
#define htols(b) (b)
#define ltohl(b) (b)
#define ltohs(b) (b)
#else
#if __BYTE_ORDER == __BIG_ENDIAN
#define htoll(b)  \
          (((((b)      ) & 0xFF) << 24) | \
	       ((((b) >>  8) & 0xFF) << 16) | \
		   ((((b) >> 16) & 0xFF) <<  8) | \
		   ((((b) >> 24) & 0xFF)      ))
#define htols(b) \
          (((((b)      ) & 0xFF) << 8) | \
		   ((((b) >> 8) & 0xFF)      ))
#define ltohl(b) htoll(b)
#define ltohs(b) htols(b)
#else
#error "Endianess not defined"
#endif
#endif

static int write_header(FILE *f)
{
	unsigned int hz=htoll(8000);
	unsigned int bhz = htoll(16000);
	unsigned int hs = htoll(16);
	unsigned short fmt = htols(1);
	unsigned short chans = htols(1);
	unsigned short bysam = htols(2);
	unsigned short bisam = htols(16);
	unsigned int size = htoll(0xFFFFFFFF);
	/* Write a wav header, ignoring sizes which will be filled in later */
	fseek(f,0,SEEK_SET);
	if (fwrite("RIFF", 1, 4, f) != 4) {
		ast_log(LOG_WARNING, "Unable to write header\n");
		return -1;
	}
	if (fwrite(&size, 1, 4, f) != 4) {
		ast_log(LOG_WARNING, "Unable to write header\n");
		return -1;
	}
	if (fwrite("WAVEfmt ", 1, 8, f) != 8) {
		ast_log(LOG_WARNING, "Unable to write header\n");
		return -1;
	}
	if (fwrite(&hs, 1, 4, f) != 4) {
		ast_log(LOG_WARNING, "Unable to write header\n");
		return -1;
	}
	if (fwrite(&fmt, 1, 2, f) != 2) {
		ast_log(LOG_WARNING, "Unable to write header\n");
		return -1;
	}
	if (fwrite(&chans, 1, 2, f) != 2) {
		ast_log(LOG_WARNING, "Unable to write header\n");
		return -1;
	}
	if (fwrite(&hz, 1, 4, f) != 4) {
		ast_log(LOG_WARNING, "Unable to write header\n");
		return -1;
	}
	if (fwrite(&bhz, 1, 4, f) != 4) {
		ast_log(LOG_WARNING, "Unable to write header\n");
		return -1;
	}
	if (fwrite(&bysam, 1, 2, f) != 2) {
		ast_log(LOG_WARNING, "Unable to write header\n");
		return -1;
	}
	if (fwrite(&bisam, 1, 2, f) != 2) {
		ast_log(LOG_WARNING, "Unable to write header\n");
		return -1;
	}
	if (fwrite("data", 1, 4, f) != 4) {
		ast_log(LOG_WARNING, "Unable to write header\n");
		return -1;
	}
	if (fwrite(&size, 1, 4, f) != 4) {
		ast_log(LOG_WARNING, "Unable to write header\n");
		return -1;
	}
	return 0;
}

/*! \brief Helper function. Get keep rec files flag*/
/*static int verbio_get_stop_on_dtmf( struct ast_speech *speech)
{
	int retval = -1;
	struct verbio_speech_info *verbio_info;
	verbio_info = (struct verbio_speech_info *)speech->data;

	if( verbio_info != NULL)
		retval = verbio_info->stop_on_dtmf;
	else
		ast_log( LOG_ERROR, "Could not get stop on dtmf.\n");
	
	return retval;
}*/

/*! \brief Helper function. Append/write samples to rec file*/
static void verbio_write_samples_to_file( struct ast_speech *speech, void *d, int l)
{
	struct verbio_speech_info *verbio_info;
	verbio_info = ( struct verbio_speech_info *)speech->data;

	if( verbio_info != NULL)
	{
		FILE *f = fopen( verbio_info->rec_file_full_path, "ab"); /*TODO: performance??...test it*/
		if( f)
		{
			if( verbio_get_sample_count( speech) == 0)
			{
				int len = 0;
				len=strlen(verbio_info->rec_file_full_path);
				if( len > 5)
					if( !strcmp((verbio_info->rec_file_full_path) + len - 4, ".wav"))
						write_header(f);
			}
			
			fwrite( d , 1 , l , f);
			fclose( f);
		}
		else
		{
			ast_log( LOG_ERROR, "Error opening rec file %s\n", 
					verbio_info->rec_file_full_path);
		}
	}

#if 0 /*TODO: use asterisk frames?*/

	/*struct ast_frame 	*f = (struct ast_frame*)malloc(sizeof(ast_frame));*/
	struct ast_frame 	*f = NULL; /*ast_frame_header_new();*/
	if (!(f = ast_calloc(1, sizeof(*f))))
		return -1;

/*
	f->mallocd_hdr_len = sizeof(*f);
	f->datalen = 0;
	f->samples = 0;
	f->data =  NULL;
	f->src = "verbio";
	f->offset = 0;
	f->mallocd=0;
	f->delivery.tv_sec = 0;
	f->delivery.tv_usec = 0;

*/
	f->offset = AST_FRIENDLY_OFFSET;
	f->frametype = AST_FRAME_VOICE;
	f->subclass = AST_FORMAT_SLINEAR;
	f->data = data;
	f->samples = len;
	f->datalen = 4;

	if( ast_writestream( s, f)) 
	{
		ast_log(LOG_WARNING, "Problem writing frame\n");
		ast_frfree(f);
	}
	ast_frfree( f);
#endif
}

#if 0
/*! \brief Helper function. Check grammar mod time */
static int verbio_check_grammar_mod( char *gram_full_path)
{

	struct ast_config 	*vcfg = NULL;	
	const char 		*vcfg_grammar_path	= NULL;

	vcfg = verbio_load_asterisk_config();
	
	if( !vcfg) 
	{
		ast_log(LOG_ERROR, "Error opening configuration file %s\n", VERBIO_CFG);
		return -1;
	}

	if( !(vcfg_grammar_path = ast_variable_retrieve(vcfg, "asr", "grammar_path"))) 
		ast_log( LOG_WARNING, "Could not read grammar_path option\n");


	int prepare_gram = 1;

	struct stat sb;
	struct stat sbtrc;
	/*char md5gram[33] = "";
	char md5test[33] = "";*/
	char md5gram[256] = "";
	char md5test[256] = "";

	if( stat( gram_full_path, &sb) == -1) 
	{
		ast_log(LOG_ERROR, "getting grammar file %s stat\n", gram_full_path);
		/*TODO: handle error.*/
	}
	else
	{
		ast_log(LOG_NOTICE, "Last (%s) mod: %s\n", gram_full_path, ctime(&sb.st_mtime));
		char trcstr[MAXPATHLEN];
		char *pch = NULL;
 		pch = strrchr( gram_full_path, '.');
		if( pch != 0)
			strncpy( trcstr, gram_full_path, pch - gram_full_path);
		else
			strcpy( trcstr, gram_full_path);

		strcat( trcstr, ".trc");
		if( stat( trcstr, &sbtrc) == -1) 
		{
			/*Could be possible... to not found the trc (grammar not yet prepared)*/
			/*ast_log(LOG_ERROR, "getting trc file %s stat\n", trcstr);*/
		}
		else
		{
			ast_log(LOG_NOTICE, "Last (%s) mod: %s\n", trcstr, ctime(&sbtrc.st_mtime));
			if( sb.st_mtime <= sbtrc.st_mtime)
			{
				prepare_gram = 0;
				ast_log(LOG_NOTICE, "Grammar (%s) does not need to be prepared.\n", gram_full_path);
			}
			else
			{
				ast_log(LOG_NOTICE, "Grammar (%s) needs to be prepared.\n", gram_full_path);	
			}
		}
		
	}

	FILE *fp;
	long len;
	char *buf;
	fp = fopen( gram_full_path,"rb");
	fseek( fp,0,SEEK_END); //go to end
	len = ftell(fp); //get position at end (length)
	fseek( fp,0,SEEK_SET); //go to beg.
	buf = (char *)malloc(len); //malloc buffer
	fread( buf,len,1,fp); //read into buffer
	fclose( fp);

	ast_log(LOG_NOTICE, "grammar contents: (%s).\n", buf);
	ast_md5_hash(md5gram, buf);
	ast_log(LOG_NOTICE, "md5sum: (%s).\n", md5gram);
	md5gram[32] = '\0';

	ast_md5_hash(md5test, "hola");
	ast_log(LOG_NOTICE, "md5sumtest: %s (hola).\n", md5test);

	char md5gram_path[MAXPATHLEN];
/*	
	char *pch = NULL;
	pch = strrchr( gram_full_path, '/');
	if( pch != 0)
		strncpy( md5gram_path, gram_full_path, pch - gram_full_path);
	else
		strcpy( md5gram_path, vcfg_grammar_path);
*/
	/* TODO: copy file to /var/lib/asterisk/verbio/gram */
	strcpy( md5gram_path, vcfg_grammar_path);
	strcat( md5gram_path, "/");
	strcat( md5gram_path, md5gram);
	ast_log(LOG_NOTICE, "copying file %s to %s\n", gram_full_path, md5gram_path);
	ast_filecopy(gram_full_path, md5gram_path, NULL);
	verbio_copy_file(gram_full_path, md5gram_path);	
	

	return prepare_gram;
}
#endif

/*! \brief Helper function. Check grammar existance in cache */
static int verbio_md5_grammar_exists( char *gram_full_path, char *md5_gram_full_path, int verbosity)
{
	struct ast_config 	*vcfg = NULL;	
	const char 		*vcfg_grammar_path	= NULL;

	vcfg = verbio_load_asterisk_config();
	
	if( !vcfg) 
	{
		ast_log(LOG_ERROR, "Error opening configuration file %s\n", VERBIO_CFG);
		return -1;
	}

	if( !(vcfg_grammar_path = ast_variable_retrieve(vcfg, "asr", "grammar_path"))) 
	{
		ast_log( LOG_ERROR, "Could not read grammar_path option from /etc/asterisk/verbio.conf. Please set it up.\n");
		ast_config_destroy( vcfg);
		return -1;
	}

	char md5gram[33] = "";
	/*char md5test[33] = "";*/
	/*char md5gram[256] = "";
	char md5test[256] = "";*/
	FILE *fp;
	long len;
	char *buf;
	size_t rres;
	char md5gram_path[MAXPATHLEN];

	strcpy(md5_gram_full_path, "");

	fp = fopen( gram_full_path,"rb");
	if( !fp)
	{
		ast_config_destroy( vcfg);
		return -1;
	}

	fseek( fp, 0, SEEK_END);
	len = ftell( fp);
	rewind( fp);
	buf = (char*) malloc( sizeof(char) * (len + 1));
	if( buf == NULL)
	{
		ast_log(LOG_ERROR, "allocating buffer.\n");
		ast_config_destroy( vcfg);
		return -1;
	}

	rres = fread( buf,1,len,fp);
	/*ast_log(LOG_NOTICE, "readed %u out of %ld bytes\n", rres, len);*/
	if( rres != len)
	{
		ast_log(LOG_ERROR, "reading grammar file to buffer.\n");
		ast_config_destroy( vcfg);
		return -1;
	}

	if( fp)
		fclose( fp);

	buf[len] = '\0';
	ast_md5_hash(md5gram, buf);
	md5gram[32] = '\0';
	/*ast_log(LOG_NOTICE, "md5sum: (%s).\n", md5gram);*/

	free( buf);

	strcpy( md5gram_path, vcfg_grammar_path);
	strcat( md5gram_path, VERBIO_GRM_CACHE_DIR);
	struct stat st;
	if( stat( md5gram_path, &st) != 0)
	{
		/* cache dir does not exists*/
		if(mkdir( md5gram_path, VERBIO_DIR_MODE) && errno != EEXIST) 
		{
			ast_log(LOG_WARNING, "mkdir '%s' failed: %s\n", md5gram_path, strerror(errno));
			ast_config_destroy( vcfg);
			return -1;
		}
		
	}

	strcat( md5gram_path, md5gram);

	/* check if file exists in  vcfg_grammar_path */
	if( access( md5gram_path, F_OK ) != -1 ) 
	{
    		/* file exists*/
		strcpy(md5_gram_full_path, md5gram_path);
		if( verbosity > 0)
			ast_log(LOG_NOTICE, "Grammar %s exists in cache (%s).\n", gram_full_path, md5_gram_full_path);

		ast_config_destroy( vcfg);

		return 1; 
	}
	else
	{
		/* Create cache file */
		verbio_copy_file(gram_full_path, md5gram_path);
		strcpy(md5_gram_full_path, md5gram_path);
	}

	ast_config_destroy( vcfg);
	return 0;
}

/* ******************************************** */
/* ************ API implementation ************ */
/* ******************************************** */
/*! \brief Find a speech recognition engine of specified name, if NULL then use the default one */
#if ASTERISK_VERSION_NUM < AST_6 
static int verbio_create( struct ast_speech *speech)
#else
static int verbio_create( struct ast_speech *speech, int format)
#endif
{
	struct verbio_speech_info *verbio_info = NULL;
	int 			dev = 0;
	struct ast_config 	*vcfg = NULL;	
	const char 		*vcfg_default_asr_config = NULL;
	const char 		*vcfg_default_asr_lang = NULL;
	const char 		*vcfg_grammar_path	= NULL;
	const char		*vcfg_recorded_files_path = NULL;
	const char		*vcfg_recorded_files_exten = NULL;
	int 			vcfg_keep_recorded_files = 0;
	int 			vcfg_mark_recorded_files = 0;
	int			vcfg_verbose		= 0;
	char 			*spkinfo = NULL;
	char 			*asrconf = NULL;
	
	const char 	*tmp = NULL;
	const char	*vcfg_vad_mode = NULL;
        float 		vcfg_max_sil  		= VERBIO_MAX_SIL;
	float 		vcfg_init_sil 		= VERBIO_INIT_SIL;
	float 		vcfg_max_ref 		= VERBIO_MAX_REF;	
	float 		vcfg_min_ref 		= VERBIO_MIN_REF;
	int 		vcfg_absolute_timeout	= VERBIO_DEF_ABS_TIMEOUT;
	float 		vcfg_low_factor		= VERBIO_LOW_FACTOR;
	float 		vcfg_high_factor	= VERBIO_HIGH_FACTOR;
	float 		vcfg_final_factor	= VERBIO_FINAL_FACTOR;
	float 		vcfg_final_high_factor	= VERBIO_FINAL_HIGH_FACTOR;
	float 		vcfg_min_high_thresh	= VERBIO_MIN_HIGH_THRESH;
	float 		vcfg_aam_min		= VERBIO_AAM_MIN;
	float		vcfg_aam_max		= VERBIO_AAM_MAX;

	char 		gram_cache_path[MAXPATHLEN];
	struct stat st;
	
	if( speech == NULL)
	{
		ast_log(LOG_ERROR, "speech is NULL\n");
		return -1;
	}

	speech->data = ast_calloc(1, sizeof(struct verbio_speech_info));
	/*ast_calloc(sizeof(struct ast_speech), 1);*/
	/*ast_calloc(1, sizeof(*verbio_info));*/
				
	/*ast_log(LOG_ERROR, "2\n");
	if( speech->data == NULL)
	{
		ast_log(LOG_ERROR, "3\n");
		speech->data = ast_calloc(sizeof(struct ast_speech), 1);
		if( speech->data == NULL)
		{
			ast_log(LOG_ERROR, "speech->data is NULL\n");
			return -1;
		}
	}
	ast_log(LOG_ERROR, "4\n");
	*/
	verbio_info = (struct verbio_speech_info *)speech->data;
	if( !verbio_info)
	{
		ast_log(LOG_ERROR, "verbio_info is NULL\n");
		return -1;
	}
	
	speech->state = AST_SPEECH_STATE_NOT_READY;

	/* *********************************** */
	/* Create and set Verbio device (port) */
	/* *********************************** */
	/* 
	  Create a unique id, in order to use it as verbio device id.
	  device = channel = port
	  Ex: if we want to attend 3 calls using recognition a the same time, we
	      need 3 different channels (each identifyed by it's own device id).
	*/
	
	/*TODO: use some kind of 'unique' id. (libuuid? custom?)*/
	dev = 1 + (int) (VERBIO_MAX_DEV * (rand() / (RAND_MAX + 1.0)));
	/*TODO: check if someone is already using this device id... how?*/

	verbio_info->device = dev;

	/* ************************************************ */
	/* Get default ASR params (readed from config file) */
	/* ************************************************ */
	vcfg = verbio_load_asterisk_config();
	
	if( !vcfg) 
	{
		ast_log(LOG_ERROR, "Error opening configuration file %s\n", VERBIO_CFG);
		return -1;
	}
	if( !(vcfg_default_asr_config = ast_variable_retrieve(vcfg, "asr", "default_config")))
	{
		ast_log( LOG_ERROR, "Error: ASR 'default_config' option missing in %s\n", VERBIO_CFG);
		goto failed;
	}
	if( !(vcfg_default_asr_lang = ast_variable_retrieve(vcfg, "asr", "default_language"))) 
	{
		ast_log( LOG_ERROR, "Error: ASR 'default_language' option missing in %s\n", VERBIO_CFG);
		goto failed;
	}

	if( !(vcfg_grammar_path = ast_variable_retrieve(vcfg, "asr", "grammar_path"))) 
	{
		ast_log( LOG_ERROR, "Could not read grammar_path option in %s\n", VERBIO_CFG);
		goto failed;
	}

	strcpy( gram_cache_path, vcfg_grammar_path);
	strcat( gram_cache_path, VERBIO_GRM_CACHE_DIR);
	if( stat( gram_cache_path, &st) != 0)
	{
		/* Cache dir does not exists...try to create it.*/
		if(mkdir( gram_cache_path, VERBIO_DIR_MODE) && errno != EEXIST) 
		{
			ast_log(LOG_ERROR, "mkdir '%s' failed: %s\n", gram_cache_path, strerror(errno));
			ast_config_destroy( vcfg);
			return -1;
		}
	}

	if( !( tmp = ast_variable_retrieve(vcfg, "debug", "verbose"))) 
		tmp = "0";
	vcfg_verbose = atoi( tmp);
	if( vcfg_verbose)
	{
		ast_log( LOG_NOTICE, "Verbose enabled in config file.\n");
		verbio_set_verbose( speech, 1);
	}

	if( (tmp = ast_variable_retrieve(vcfg, "debug", "keep_recorded_files"))) 
		vcfg_keep_recorded_files = atoi( tmp);

	if( (tmp = ast_variable_retrieve(vcfg, "debug", "mark_recorded_files"))) 
		vcfg_mark_recorded_files = atoi( tmp);
	
	vcfg_recorded_files_path = ast_variable_retrieve(vcfg, "debug", "recorded_files_path"); 
				
	vcfg_recorded_files_exten = ast_variable_retrieve(vcfg, "debug", "recorded_files_exten"); 
	  
	if( (tmp = ast_variable_retrieve(vcfg, "asr", "init_sil"))) 
		vcfg_init_sil = atof( tmp);

	if( (tmp = ast_variable_retrieve(vcfg, "asr", "max_sil"))) 
		vcfg_max_sil = atof( tmp);
	
	if( (tmp = ast_variable_retrieve(vcfg, "asr", "max_ref"))) 
		vcfg_max_ref = atof( tmp);
	
	if( (tmp = ast_variable_retrieve(vcfg, "asr", "min_ref"))) 
		vcfg_min_ref = atof( tmp);

	if( (tmp = ast_variable_retrieve(vcfg, "asr", "absolute_timeout"))) 
		vcfg_absolute_timeout = atoi( tmp);

	if( !(vcfg_vad_mode = ast_variable_retrieve(vcfg, "vad", "mode")))
	{

		if( (tmp = ast_variable_retrieve(vcfg, "vad", "low_factor"))) 
			vcfg_low_factor = atof( tmp);
	
		if( (tmp = ast_variable_retrieve(vcfg, "vad", "high_factor"))) 
			vcfg_high_factor = atof( tmp);
	
		if( (tmp = ast_variable_retrieve(vcfg, "vad", "final_factor"))) 
			vcfg_final_factor = atof( tmp);
	
		if( (tmp = ast_variable_retrieve(vcfg, "vad", "final_high_factor"))) 
			vcfg_final_high_factor = atof( tmp);
	
		if( (tmp = ast_variable_retrieve(vcfg, "vad", "min_high_thresh"))) 
			vcfg_min_high_thresh = atof( tmp);
	
		if( (tmp = ast_variable_retrieve(vcfg, "vad", "aam_min"))) 
			vcfg_aam_min = atof( tmp);
	
		if( (tmp = ast_variable_retrieve(vcfg, "vad", "aam_max"))) 
			vcfg_aam_max = atof( tmp);
	}

	/* ******************************** */
	/* Set previously readed ASR params */
	/* ******************************** */
	if( verbio_set_curr_config( speech, vcfg_default_asr_config) < 0)
		goto failed;
	
	if( verbio_set_curr_lang( speech, vcfg_default_asr_lang) < 0)
		goto failed;

	if( vcfg_grammar_path) 
		verbio_set_def_gram_path( speech, vcfg_grammar_path);

	if( verbio_set_abs_timeout( speech, vcfg_absolute_timeout) < 0)
		goto failed;

	if( verbio_set_curr_vad_param( speech, VVAD_MAX_SIL, vcfg_max_sil) < 0)
		goto failed;
      			
	if( vcfg_vad_mode)
		verbio_set_vad_mode(speech, vcfg_vad_mode);
	else
	{	
		if( verbio_set_curr_vad_param( speech, VVAD_INIT_SIL, vcfg_init_sil) < 0)
			goto failed;	
		
		if( verbio_set_curr_vad_param( speech, VVAD_MAX_REF, vcfg_max_ref) < 0)
			goto failed;
		
		if( verbio_set_curr_vad_param( speech, VVAD_MIN_REF, vcfg_min_ref) < 0)
			goto failed;
			
		if( verbio_set_curr_vad_param( speech, VVAD_LOW_FACTOR, vcfg_low_factor) < 0)
			goto failed;
		
		if( verbio_set_curr_vad_param( speech, VVAD_HIGH_FACTOR, vcfg_high_factor) < 0)
			goto failed;
		
		if( verbio_set_curr_vad_param( speech, VVAD_FINAL_FACTOR, vcfg_final_factor) < 0)
			goto failed;
		
		if( verbio_set_curr_vad_param( speech, VVAD_FINAL_HIGH_FACTOR, vcfg_final_high_factor) < 0)
			goto failed;
		
		if( verbio_set_curr_vad_param( speech, VVAD_MIN_HIGH_THRESH, vcfg_min_high_thresh) < 0)
			goto failed;
		
		if( verbio_set_curr_vad_param( speech, VVAD_AAM_MIN, vcfg_aam_min) < 0)
			goto failed;
		
		if( verbio_set_curr_vad_param( speech, VVAD_AAM_MAX, vcfg_aam_max) < 0)
			goto failed;
	}

	if( vcfg_keep_recorded_files)
	{
		if( vcfg_recorded_files_path)
			verbio_set_keep_rec_files( speech, 1);
		else
		{
			verbio_set_keep_rec_files( speech, 0);
			ast_log(LOG_WARNING, "Not going to keep recorded files. recorded_files_path not set\n");	
		}
	}
	else
		verbio_set_keep_rec_files( speech, 0);

	if( vcfg_mark_recorded_files)
	{
		if( vcfg_keep_recorded_files)
			verbio_set_mark_rec_files( speech, 1);
		else
		{
			verbio_set_mark_rec_files( speech, 0);
			ast_log(LOG_WARNING, "Not going to mark rec files. keep_recorded_files not set\n");	
		}
	}
	else
		verbio_set_mark_rec_files( speech, 0);

	verbio_set_stop_on_dtmf( speech, 0); /* 0 => by default,do not stop on dtmf detection*/

	
	
	if( verbio_get_verbose( speech) > 0)
	{
		if( vox_getparm(-1, VXGB_TTSSPKINFO, &spkinfo) == 0)
			if( spkinfo)
				ast_log(LOG_NOTICE,"Available TTS speakers: %s\n", spkinfo);
	
		if( vox_getparm(-1, VXGB_START_CONF, &asrconf) == 0)
			if( asrconf)
				ast_log(LOG_NOTICE,"Available ASR configs: %s\n\n", asrconf);

		ast_log(LOG_NOTICE, "---- verbio_create going to set: ----\n");
		ast_log(LOG_NOTICE, "- Verbio Device: %d\n", 
					dev);
		ast_log(LOG_NOTICE, "- ASR config: %s\n", 
					vcfg_default_asr_config);
		ast_log(LOG_NOTICE, "- ASR lang: %s\n", 
					vcfg_default_asr_lang);
		if( vcfg_grammar_path)
			ast_log(LOG_NOTICE, "- Def gram path: %s\n", 
							vcfg_grammar_path);
		else
			ast_log(LOG_WARNING, "- grammar_path not defined. Check verbio.conf !\n");

		ast_log(LOG_NOTICE, "- init sil: %f\n", vcfg_init_sil);
		ast_log(LOG_NOTICE, "- max sil: %f\n", vcfg_max_sil);
		ast_log(LOG_NOTICE, "- absolute timeout: %d\n", 
						vcfg_absolute_timeout);
		
#if 0
		if( !vcfg_vad_mode)
		{
			ast_log(LOG_NOTICE, "- max ref: %f\n", vcfg_max_ref);
			ast_log(LOG_NOTICE, "- min ref: %f\n", vcfg_min_ref);
			ast_log(LOG_NOTICE, "- absolute timeout: %d\n", vcfg_absolute_timeout);
			ast_log(LOG_NOTICE, "- low factor: %f\n", vcfg_low_factor);
			ast_log(LOG_NOTICE, "- high factor: %f\n", vcfg_high_factor);
			ast_log(LOG_NOTICE, "- final factor: %f\n", vcfg_final_factor);
			ast_log(LOG_NOTICE, "- final high factor: %f\n", vcfg_final_high_factor);
			ast_log(LOG_NOTICE, "- min high thresh: %f\n", vcfg_min_high_thresh);
			ast_log(LOG_NOTICE, "- aam min: %f\n", vcfg_aam_min);
			ast_log(LOG_NOTICE, "- aam max: %f\n", vcfg_aam_max);
		}
#endif
		
		if( vcfg_vad_mode)
			ast_log(LOG_NOTICE, "- VAD mode: %s\n",  vcfg_vad_mode);
		else
			ast_log(LOG_NOTICE, "- VAD params:\n");
		
		ast_log(LOG_NOTICE, "\t-(srv) max ref: %f\n", verbio_info->max_ref);
		ast_log(LOG_NOTICE, "\t-(srv) min ref: %f\n", verbio_info->min_ref);
		ast_log(LOG_NOTICE, "\t-(cli) low factor: %f\n", verbio_info->low_factor);
		ast_log(LOG_NOTICE, "\t-(cli) high factor: %f\n", verbio_info->high_factor);
		ast_log(LOG_NOTICE, "\t-(cli) final factor: %f\n", verbio_info->final_factor);
		ast_log(LOG_NOTICE, "\t-(cli) final high factor: %f\n", verbio_info->final_high_factor);
		ast_log(LOG_NOTICE, "\t-(cli) min high thresh: %f\n", verbio_info->min_high_thresh);
		ast_log(LOG_NOTICE, "\t-(cli) aam min: %f\n", verbio_info->aa_min);
		ast_log(LOG_NOTICE, "\t-(cli) aam max: %f\n", verbio_info->aa_max);


		if( vcfg_keep_recorded_files)
		{
			if( vcfg_recorded_files_path)	
			{			
				ast_log(LOG_NOTICE, "- Rec files path: %s\n", 
						vcfg_recorded_files_path);
				
				if( vcfg_recorded_files_exten)
					ast_log(LOG_NOTICE, "- Rec file exten: %s\n", vcfg_recorded_files_exten);
				else
					ast_log(LOG_NOTICE, "- Rec file exten: %s\n", exten);

				if( vcfg_mark_recorded_files)
					ast_log(LOG_NOTICE, "- Mark rec file: yes\n");
			}
			else
			{
				ast_log(LOG_WARNING, "- recorded_files_path not set.\n");
			}
		}  
		ast_log(LOG_NOTICE, "-------------------------------------\n");	
	}

	verbio_incr_sample_count( speech, -1); /* -1 => Set sample count to 0*/
	
	ast_config_destroy( vcfg);
	return 0;

 failed:
	ast_config_destroy( vcfg);
	return -1;
}

/*! \brief  Destroy connection to engine. */
static int verbio_destroy( struct ast_speech *speech)
{
	int verbose = verbio_get_verbose( speech);
	struct verbio_speech_info *verbio_info;
	/*struct 	ast_speech_result *result = NULL;*/
	int dev = 0;

	if( speech == NULL)
	{
		ast_log(LOG_ERROR, "verbio_destroy speech struct is NULL!\n");
		return -1;
	}

	verbio_info = (struct verbio_speech_info *)speech->data;
	
	/*TODO: free structures (results, grammar info,...)*/
	if( verbio_info != NULL)
	{
		/* Free dev resources...otherwise we will run out of lines*/
		dev = verbio_get_curr_dev( speech);
		vox_devclose( dev); 

		/*result = speech->results;
		result = result->next;
		while( result)
		{
			ast_free( result);
			result = result->next;
		}*/

		/*ast_free( speech->results);
		ast_free( result);*/

		ast_free( verbio_info);
	}

	/*ast_log(LOG_NOTICE, "ast_free( speech)\n");
	ast_free( speech);*/
	speech->data = NULL;

	if( verbose > 0)
		ast_log(LOG_NOTICE, "verbio_destroy connection closed (%d)\n", dev);
	
	return 0;
}

/*! \brief Load a local grammar on a speech structure */
static int verbio_load(struct ast_speech *speech, char *grammar_name, char *grammar)
{
	int 		verbose = verbio_get_verbose( speech);
	const char 	*asr_config = verbio_get_curr_config( speech);
	const char 	*asr_lang = verbio_get_curr_lang( speech);
	int 		dev = verbio_get_curr_dev( speech);
	char 		gram_full_path[MAXPATHLEN];
	char 		gram_cache_full_path[MAXPATHLEN];
	int 		nind = 0;
	int 		mode = verbio_get_grammar_mode( speech, 
						(const char*)grammar);
	int cache_grammar_exists = 0;
	
	if( !asr_config || !asr_lang || dev == -2)
	{
		ast_log(LOG_ERROR, "Fatal error in verbio_load\n");
		return -1;
	}

	if( verbio_get_grammar_id( speech, grammar_name) != -2)
	{
		ast_log(LOG_WARNING, "Grammar %s already loaded\n", 
							grammar_name);
		return 0;
	}	

	/* <Grammar file> */
	if( grammar[0] != '/')
	{
		/* Full path not provided, try to read from default gram path.
		   (default grammar path readed from verbio.conf)
		*/
		const char* def_grm_path = verbio_get_def_gram_path( speech);
		if( !def_grm_path)
		{
			ast_log(LOG_ERROR, "Error: default grammar path missing\n");	
			return -1;
		}

		strcpy( gram_full_path, def_grm_path); /*FIXME: ast_copy_string? (if exists, use it)*/
		strcat( gram_full_path, "/");	
		strcat( gram_full_path, grammar);	
	}
	else
	{
		strcpy( gram_full_path, grammar);
	}

	
	if( vox_setparm( dev,VXCH_DEFASRCFG, asr_config) == -1)
	{
		ast_log( LOG_ERROR, "Error: Can not set ASR config to %s [%s]\n", 
							asr_config, 
							ATVOX_ERRMSGP( dev));
		return -1;
	}

	if( vox_setparm( dev,VXCH_DEFASRLANG, asr_lang) == -1)
	{
		ast_log( LOG_ERROR, "Error: Can not set ASR lang to %s [%s]\n", 
							asr_lang, 
							ATVOX_ERRMSGP( dev));
		return -1;
	}

	/* Check grammar modification */
	//grammar_modified = verbio_check_grammar_mod( gram_full_path); 
	cache_grammar_exists = verbio_md5_grammar_exists( gram_full_path, gram_cache_full_path, verbose);

	if( (verbose > 0) && (cache_grammar_exists > 0))
		ast_log( LOG_NOTICE, "No need to prepare grammar %s (cache: %s).\n", grammar, gram_cache_full_path);
 	else
	{
		/* md5sum file created in verbio_md5_grammar_exists function */
		if( verbose > 0)
			ast_log( LOG_NOTICE, "Preparing grammar %s conf:%s lang:%s. Creating cache file: %s\n", 
									grammar, 
									asr_config, 
									asr_lang,
									gram_cache_full_path);
	
		if( vox_prevcbdev( dev, gram_cache_full_path, mode, &nind, asr_lang) == -1) 
		{
			
			ast_log(LOG_ERROR, "Error preparing grammar [%s line: %d]\n", 
							ATVOX_ERRMSGP( dev), nind);
			ast_log(LOG_ERROR, "- grammar path: %s\n", grammar);
			ast_log(LOG_ERROR, "- cache grammar path: %s\n", gram_cache_full_path);
		
			return -1;
		}
	}

	nind = vox_loadvcb( dev, gram_cache_full_path, mode);
	if( nind == -1) 
	{
		ast_log(LOG_ERROR, "Error loading grammar [%s line %d]\n",
						ATVOX_ERRMSGP( dev), nind);
		ast_log(LOG_ERROR, "- grammar: %s\n", grammar);
		ast_log(LOG_ERROR, "- cache: %s\n", gram_cache_full_path);

		return -1;
	}
	
	if( verbio_add_grammar( speech, grammar_name, gram_full_path, nind) < 0)
	{
		ast_log(LOG_ERROR, "Error: grammar %s (cache: %s) not added to list\n", 
							grammar_name, gram_cache_full_path);
		return -1;
	}

	return 0;
}

/*! \brief Unload a local grammar from a speech structure */
static int verbio_unload(struct ast_speech *speech, char *grammar_name)
{
	int dev = verbio_get_curr_dev( speech);
	int gid = verbio_get_grammar_id( speech, grammar_name);
	
	if( dev == -2)
		return -1;

#if 0
	if( gid == -1)/* TODO allow this? (to unload all grammars)*/
	{
		if( vox_setvcb( dev, NULL, 0))/* Free licences */
			ast_log( LOG_ERROR, "vox_setvcb. [%s]\n", ATVOX_ERRMSGP( dev));
		
		vox_devclose( dev); /* Free dev resources...otherwise we will run out of lines*/
	}
#endif

	if( gid != -2)/*get_grammar_id returns -2 when grammar not found*/
	{
		if( vox_unloadvcb( dev, gid, 0) < 0) 
		{
			ast_log( LOG_ERROR, "Error unloading grammar %s [%s]\n", 
							grammar_name, 
							ATVOX_ERRMSGP( dev));
			/*return -1;*/
		}
		/*remove grammar from list*/
		verbio_remove_gram( speech, gid);
	}
	else
	{
		ast_log( LOG_WARNING, "Grammar %s not found\n", grammar_name);
	}
	return 0;
}

/*! \brief Activate a loaded (either local or global) grammar */
static int verbio_activate(struct ast_speech *speech, char *grammar_name)
{
	/* Activate grammar */
	int dev = verbio_get_curr_dev( speech);
	int gid = verbio_get_grammar_id( speech, grammar_name);
	struct verbio_speech_info *verbio_info;
	const char *gram_path = NULL;
	int mode = GVX_ABNF;

	verbio_info = (struct verbio_speech_info *)speech->data;
	if( verbio_info == NULL)
	{
		ast_log(LOG_ERROR, "Error: grammar %s Not found  \n", grammar_name);
		return 0;/*TODO: return -1 ??*/
	}

	gram_path = verbio_info->grammar[gid].path;
	mode = verbio_get_grammar_mode( speech, gram_path);

	/*TODO: allow this situation? (repeated grammars)*/
	/*get_grammar_id returns -1 when grammar not found*/
	/*while( gid != -1) 
	{*/
	if( gid == -2)/*get_grammar_id returns -2 when grammar not found*/
	{
		ast_log(LOG_ERROR, "Error: grammar %s not found\n", grammar_name);
		return 0;/*TODO: return -1 ??*/
	}

	if( vox_activatevcb( dev, gid, mode) == -1) 
	{
		ast_log(LOG_ERROR, "Error activating grammar [%s]\n", 
							ATVOX_ERRMSGP( -1));
		return -1;
	}
	
	/*}*/
	return 0;
}

/*! \brief Deactivate a loaded grammar on a speech structure */
static int verbio_deactivate(struct ast_speech *speech, char *grammar_name)
{
	/* Deactivate grammar */
	int dev = verbio_get_curr_dev( speech);
	int gid = verbio_get_grammar_id( speech, grammar_name);
	struct verbio_speech_info *verbio_info;
	const char *gram_path = NULL;
	int mode = GVX_ABNF;

	verbio_info = (struct verbio_speech_info *)speech->data;
	if( verbio_info == NULL)
	{
		ast_log(LOG_ERROR, "Could not deactivate grammar %s\n", 
								grammar_name);
		return 0;/*TODO: return -1 ??*/
	}

	gram_path = verbio_info->grammar[gid].path;
	if( !gram_path)
		return 0; /*TODO: return -1 ??*/

	mode = verbio_get_grammar_mode( speech, gram_path);
	/*int mode = verbio_get_grammar_mode( grammar_name);*/

	/*deactivate (maybe) repeated grammars*/
	/*while( gid != -2) 
	{*/
	if( gid != -2) /*get_grammar_id returns -2 when grammar not found*/
	{
		if( vox_deactivatevcb( dev, gid, mode) == -1) 
		{
			ast_log(LOG_ERROR, "Error deactivating grammar [%s]\n", 
							ATVOX_ERRMSGP( -1));
			return 0; /*TODO: return -1?*/
		}
	}
	else
	{
		ast_log( LOG_WARNING, "Grammar %s not found\n", grammar_name);
	}
	/*
		gid = verbio_get_grammar_id( speech, grammar_name);
	}
	*/
	return 0;
}

/*! \brief Write in signed linear audio to be recognized */
static int verbio_write(struct ast_speech *speech, void *data, int len)
{
  	unsigned short max;

	/* Pass samples to Voice Activity Detector or ASR engine.
	   Depending on voice detection status.
	*/
	if( verbio_get_keep_rec_files( speech) > 0)
		verbio_write_samples_to_file( speech, data, len);
		
	if( verbio_get_voice_detected( speech) != 1) /*User has not spoken yet*/
	{
		/* 
		  		= VAD Client-side = 
			Please edit /etc/asterisk/verbio.conf (vad section)
			if you want to control vad sensitivity.
		*/
		int status = 0;
		VAD_PARAM *vsddev = verbio_get_vsddev( speech);
		status = vox_vsd_write( vsddev, data, len);
		if( (status != VVX_INIT) && (status != VVX_SILENCE))
		{
			max = 32000;
	    		if( verbio_get_mark_rec_files( speech) > 0)			
				verbio_write_samples_to_file( speech, (void*)&max, 2);

			/* Voice detected */
			speech->flags = AST_SPEECH_QUIET;/*Quiet prompt.*/
			verbio_set_voice_detected( speech, 1);
			if( verbio_get_verbose( speech) > 0)
				ast_log(LOG_NOTICE, "Voice detected\n");
				
		}

		/* VAD (Voice Activity Detection) Server-side*/
		/* Best on client-side (do not bother ASR engine with VAD)*/
		/*vox_getparm( dev, VXCH_VOICEDETECTED, &voice_detected);*/
		/*if( voice_detected == 1)
			ast_stopstream( chan);*/
	}
 				
	if( verbio_get_voice_detected( speech) == 1) /* Speech has been detected. 
							Now, we need to send samples to ASR engine.*/
	{
		int recdev = verbio_get_recdev( speech);
		if( vox_recstr_write( recdev, data, len) != len)
		{
			max = -32000;
	    		if( verbio_get_mark_rec_files( speech) > 0)			
				verbio_write_samples_to_file( speech, (void*)&max, 2);

			if( verbio_get_verbose( speech) > 0)
				ast_log(LOG_NOTICE, "Set Result\n");

			return verbio_set_asr_result( speech);
		}
	}

	/* Increase processed samples counter*/
	verbio_incr_sample_count( speech, len);

	/* absolute timeout check*/
	unsigned long int sample_count = verbio_get_sample_count( speech);
	unsigned long int fs_x_t = verbio_get_fs_x_tout( speech);
	if( sample_count < 0 || fs_x_t < 0)
		return -1;

	if( sample_count >= fs_x_t)
	{
		ast_log(LOG_WARNING, "Absol timeout reached (%ld)\n", 
								sample_count);
		speech->flags = AST_SPEECH_SPOKE; 
		speech->state = AST_SPEECH_STATE_DONE;
		verbio_incr_sample_count( speech, -1); /*reset sample count*/
	}

	return 0;
}

/*! \brief Signal to the engine that DTMF was received */
static int verbio_dtmf(struct ast_speech *speech, const char *dtmf)
{
	/*TODO: implementation??*/
	ast_log(LOG_NOTICE, "Verbio dtmf not implemented\n");
	return 0;
}

/*! \brief Start speech recognition on a speech structure */
static int verbio_start(struct ast_speech *speech)
{
	struct ast_speech_engine *verbio_engine;
	int recdev = 0; /*recdev inited in this function*/
	VAD_PARAM *vsddev = NULL; /* vsddev inited in this function*/
	float 	init_sil = verbio_get_curr_vad_param( speech, VVAD_INIT_SIL);
	float 	max_sil =verbio_get_curr_vad_param( speech, VVAD_MAX_SIL);;
	VX_RSP 	rsp;
	float low_factor = verbio_get_curr_vad_param( speech, VVAD_LOW_FACTOR);
	float high_factor = verbio_get_curr_vad_param( speech, VVAD_HIGH_FACTOR);
	float final_factor = verbio_get_curr_vad_param( speech, VVAD_FINAL_FACTOR);
	float final_high_factor = verbio_get_curr_vad_param( speech, VVAD_FINAL_HIGH_FACTOR);
	float min_high_thresh = verbio_get_curr_vad_param( speech, VVAD_MIN_HIGH_THRESH);
	float aam_min = verbio_get_curr_vad_param( speech, VVAD_AAM_MIN);
	float aam_max = verbio_get_curr_vad_param( speech, VVAD_AAM_MAX);
	float min_ref = verbio_get_curr_vad_param( speech, VVAD_MIN_REF);
	float max_ref = verbio_get_curr_vad_param( speech, VVAD_MAX_REF);
	VAD_PRM prm;

	int dev = verbio_get_curr_dev( speech);
	
	if( dev == -2 || /* -1 reserved (to specify all channels)*/
			init_sil < 0 || 
			max_sil < 0 ||
			low_factor < 0 || 
			high_factor < 0 || 
			final_factor < 0 || 
			final_high_factor < 0 || 
			min_high_thresh < 0 || 
			aam_min < 0 ||
			aam_max < 0 ||
			min_ref < 0 ||
			max_ref < 0)
	{
		ast_log(LOG_ERROR, "Error: init params\n");
		return -1;
	}

	verbio_engine = speech->engine;

	/* ***** rec device ***** */
	vox_clrrsp( &rsp);
	rsp.maxsil = max_sil;
	rsp.initsil = init_sil;

	if( (recdev = vox_recstr_open( dev, &rsp, VERBIO_CODEC | MC_INITSIL )) == -1) 
	{
		ast_log(LOG_ERROR, "Error: vox_recstr_open [%s]\n", ATVOX_ERRMSGP( dev));
		return -1;	
	}

	/* Store recdev in speech struct. We will need recdev later (when 
	   sending audio samples to recognizer)
	*/
	verbio_set_recdev( speech, recdev);

	/* ****** VAD (Server-side) ****** */
	if( min_ref >= max_ref)
	{
		min_ref = max_ref * 0.5;
		ast_log( LOG_WARNING, "(min_ref >= max_ref). Setting min_ref to: %f\n", min_ref);
	}

	if( vox_setparm( dev, VXGB_VSDMINREF, &min_ref) == -1)
			ast_log( LOG_ERROR, "Error setting min_ref.\n");

	if( vox_setparm( dev, VXGB_VSDMINREF, &max_ref) == -1)
			ast_log( LOG_ERROR, "Error setting max_ref.\n");

	/* ***** VAD (Client-side) ***** */
	vox_clrvad( &prm);
	prm.low_factor = low_factor;
	prm.high_factor = high_factor;
	prm.final_factor = final_factor;
	prm.final_high_factor = final_high_factor;
	prm.min_high_thresh = min_high_thresh;
	prm.aam_min = aam_min;
	prm.aam_max = aam_max;
	
	vsddev = vox_vsd_open( dev, recdev, &prm, VERBIO_CODEC);
	if( (int)vsddev == -1)
	{
		ast_log(LOG_ERROR, "Error: vsddev could not be opened\n");
		vox_recstr_close( dev, recdev);
		return -1;
	}

	/* Store vsddev in speech struct. We will need vsddev later (when
	   doing VAD.)
	*/
	verbio_set_vsddev( speech, vsddev);

	/* Init voice detection status*/
	verbio_set_voice_detected( speech, 0);

	/* set rec file (to record user input) */
	if( verbio_get_keep_rec_files( speech) > 0)
		if( verbio_set_rec_file( speech) < 0)
			ast_log(LOG_ERROR, "Error setting rec file.\n");

	/* Verbio engine is ready to accept samples */
	speech->state = AST_SPEECH_STATE_READY;

	return 0;
}

/*! \brief Change an engine specific attribute */
static int verbio_change(struct ast_speech *speech, char *name, const char *value)
{
	int verbose = verbio_get_verbose( speech);
	int retval = 0;

	if( verbose > 0 )
		ast_log(LOG_NOTICE, "Setting attribute \'%s\' to \'%s\'\n", 
									name, 
									value);
	if( !strcmp( name, "verbose"))
	{
		verbio_set_verbose( speech, atoi( value));
	}
	else if( !strcmp( name, "config"))
	{
		if( verbio_set_curr_config( speech, value) < 0)
			retval = -1;
	}
	else if( !strcmp( name, "language"))
	{
		if( verbio_set_curr_lang( speech, value) < 0)
			retval = -1;
	}
	else if( !strcmp( name, "lang"))
	{
		if( verbio_set_curr_lang( speech, value) < 0)
			retval = -1;
	}
	else if( !strcasecmp( name, "languageiso"))
	{
		if( verbio_set_curr_lang_conf( speech, value) < 0)
			retval = -1;
	}
	else if( !strcasecmp( name, "langiso"))
	{
		if( verbio_set_curr_lang_conf( speech, value) < 0)
			retval = -1;
	}
	/*else if( !strcmp( name, "global_grammar"))
	{
		ast_log(LOG_NOTICE, "Setting global grammar %s.\n", value);
		if( verbio_set_global_grammar( speech, value) < 0)
			retval = -1;
	}*/
	else if( !strcmp( name, "mode"))
	{
		if( verbio_set_vad_mode(speech, name) < 0)
			retval = -1;
	}
	else if( !strcmp( name, "init_sil"))
	{
		if( verbio_set_curr_vad_param( speech, VVAD_INIT_SIL, atof(value)) < 0)
			retval = -1;
	}
	else if( !strcmp( name, "max_sil"))
	{
		if( verbio_set_curr_vad_param( speech, VVAD_MAX_SIL, atof(value)) < 0)
			retval = -1;
	}
	else if( !strcmp( name, "min_ref"))
	{
		if( verbio_set_curr_vad_param( speech, VVAD_MIN_REF, atof(value)) < 0)
			retval = -1;
	}
	else if( !strcmp( name, "max_ref"))
	{
		if( verbio_set_curr_vad_param( speech, VVAD_MAX_REF, atof(value)) < 0)
			retval = -1;
	}
	else if( !strcmp( name, "absolute_timeout"))
	{
		if( verbio_set_abs_timeout( speech, atoi(value)) < 0) 
			retval = -1;
	}
	else if( !strcmp( name, "low_factor"))
	{
		if( verbio_set_curr_vad_param( speech, VVAD_LOW_FACTOR, atof(value)) < 0)
			retval = -1;
	}
	else if( !strcmp( name, "high_factor"))
	{
		if( verbio_set_curr_vad_param( speech, VVAD_HIGH_FACTOR, atof(value)) < 0)
			retval = -1;	
	}
	else if( !strcmp( name, "final_factor"))
	{
		if( verbio_set_curr_vad_param( speech, VVAD_FINAL_FACTOR, atof(value)) < 0)
			retval = -1;	
	}
	else if( !strcmp( name, "final_high_factor"))
	{
		if( verbio_set_curr_vad_param( speech, VVAD_FINAL_HIGH_FACTOR, atof(value)) < 0)
			retval = -1;	
	}
	else if( !strcmp( name, "min_high_thresh"))
	{
		if( verbio_set_curr_vad_param( speech, VVAD_MIN_HIGH_THRESH, atof(value)) < 0)
			retval = -1;	
	}
	else if( !strcmp( name, "aam_min"))
	{
		if( verbio_set_curr_vad_param( speech, VVAD_AAM_MIN, atof(value)) < 0)
			retval = -1;	
	}
	else if( !strcmp( name, "aam_max"))
	{
		if( verbio_set_curr_vad_param( speech, VVAD_AAM_MAX, atof(value)) < 0)
			retval = -1;	
	}
	else if( !strcmp( name, "keep_recorded_files"))
	{
		verbio_set_keep_rec_files( speech, atoi(value));
	}
	else if( !strcmp( name, "other_atrribute"))
	{
		ast_log(LOG_NOTICE, "Setting other_atrributee to %s\n", value);
	}
	else
	{
		ast_log(LOG_WARNING, "Unknown attribute %s\n", name);
	}

	if( retval < 0)
		ast_log(LOG_ERROR, "Error setting %s to %s\n", name, value);

	return retval;
}

/*! \brief  Change the type of results we want back  */
static int verbio_change_results_type(struct ast_speech *speech, enum ast_speech_results_type results_type)
{
	/*TODO: implementation */
	ast_log(LOG_NOTICE, "Verbio change results type not available yet\n");
	return 0;
}

/*! \brief Try to get results */
static struct ast_speech_result* verbio_get(struct ast_speech *speech)
{
	/*verbio_set_asr_result( speech);*/
	return speech->results;
}

struct ast_speech_engine verbio_engine = {
	.name = "verbio",
	.create = verbio_create,
	.destroy = verbio_destroy,
	.load = verbio_load,
	.unload = verbio_unload,
	.activate = verbio_activate,
	.deactivate = verbio_deactivate,
	.write = verbio_write,
	.dtmf = verbio_dtmf,
	.start = verbio_start,
	.change = verbio_change,
	.change_results_type = verbio_change_results_type,
	.get = verbio_get,
	.formats = AST_FORMAT_SLINEAR,
};

static int load_module(void)
{
	ast_log(LOG_NOTICE, "Loading Verbio Speech Technologies res speech\n");

	if( init_verbio_res_speech() == 1)
		return ast_speech_register(&verbio_engine);
	else 
		return 0;/*do not stop asterisk startup*/
}

static int unload_module(void)
{
	ast_log(LOG_NOTICE, "Unloading Verbio Speech Technologies res speech\n");
	close_verbio_res_speech();
	sleep( 2); /*be careful with (fast) reconnections*/

	return ast_speech_unregister(verbio_engine.name);
}

#undef AST_BUILDOPT_SUM
#define AST_BUILDOPT_SUM ""

AST_MODULE_INFO_STANDARD(ASTERISK_GPL_KEY, "Verbio Speech Technologies ASR engine");

