#include <LV_SRE.h>
#include <iostream>

#include <string>
#include <stdlib.h>
#include <stdio.h>

using namespace std;

// Uncomment the next line to use the C++ implementation (Commented = C style)
#define USE_CPP_API
#ifdef USE_CPP_API
#include <LVSpeechPort.h>
#endif

void PrintUsage(const char *prog_name)
{
	printf("Usage: %s <grammar-label> <grammar-file> <sound-format> <audio-file> [server-ip]\n", prog_name);
	printf("    where <sound-format> is one of:\n");
	printf("      0	for Unknown Format\n");
	printf("      1	for ULAW  8 kHz\n");
	printf("      2	for PCM   8 kHz\n");
	printf("      3	for PCM  16 kHz\n");
	printf("      4	for ALAW  8 kHz\n");
}

int main(int argc, char *argv[])
{
	// Handle command line inputs
	if(argc < 5 || argc > 6)
	{
		PrintUsage(argv[0]);

		return 1;
	}

	int snd_frmt = atoi(argv[3]);
	
	if(snd_frmt < 0 || snd_frmt > 4)
	{
		PrintUsage(argv[0]);
	
		return 1;
	}

	SOUND_FORMAT audio_format = static_cast<SOUND_FORMAT>(snd_frmt);

	string grammar_label = argv[1],
		grammar_file = argv[2],
		audio_file = argv[4];

	unsigned char* audio_buffer;

	int vc = 1;

	int retval, exit_code = 0;

	if(argc == 6)
	{
		LV_SRE_SetPropertyEx(NULL, PROP_EX_SRE_SERVERS, PROP_EX_VALUE_TYPE_STRING, argv[5], PROP_EX_TARGET_CLIENT, 0);
	}
	
	// Read Audio
	FILE *AUDIO = fopen(audio_file.c_str(), "rb");
	
	if (!AUDIO)
	{
		printf("Error opening audio file : %s\n", audio_file.c_str());
	
		return 2;
	}

	fseek(AUDIO, 0, SEEK_END);

	int flength = ftell(AUDIO);

	rewind(AUDIO);

	audio_buffer = new unsigned char[flength + 1];

	fread(audio_buffer, flength, 1, AUDIO);

	fclose(AUDIO);

	// 1. Open Port
	int OpenErrorCode = 0;

#ifdef USE_CPP_API

	LVSpeechPort port;

	port.CreateClient();

	//=========================================================================
	//Speech Port setup
	retval = port.SetPropertyEx(PROP_EX_SAVE_SOUND_FILES, PROP_EX_VALUE_TYPE_INT, (void *)1, PROP_EX_TARGET_PORT, 0);

	if (retval != LV_SUCCESS)
	{
		printf("%d (%s)", retval, port.ReturnErrorString(retval));
	}	

	retval = port.SetPropertyEx(PROP_EX_MAX_NBEST_RETURNED, PROP_EX_VALUE_TYPE_INT, (void*)10, PROP_EX_TARGET_PORT, 0);

	if (retval != LV_SUCCESS)
	{
		printf("%d (%s)", retval, port.ReturnErrorString(retval));
	}

	retval = port. SetPropertyEx(PROP_EX_DECODE_TIMEOUT, PROP_EX_VALUE_TYPE_INT, (void*)100000, PROP_EX_TARGET_PORT, 0);

	if (retval != LV_SUCCESS)
	{
		printf("%d (%s)", retval, port.ReturnErrorString(retval));
	}

	//=========================================================================
	//Load data into port
	retval = port.LoadGrammar(grammar_label.c_str(), grammar_file.c_str());

	if(retval == LV_SUCCESS)
	{
		printf("Loaded Grammar!\n");

		retval = port.ActivateGrammar(grammar_label.c_str());

		if(retval == LV_SUCCESS)
		{
			retval = port.LoadVoiceChannel(vc, audio_buffer, flength, audio_format);

			if(retval == LV_SUCCESS)
			{
				printf("Loaded Audio!\n");

				retval = port.Decode(vc, LV_ACTIVE_GRAMMAR_SET, LV_DECODE_BLOCK | LV_DECODE_SEMANTIC_INTERPRETATION);

				if (retval < 0)
				{
					printf("Decode failed with error code %d (%s)\n", retval, port.ReturnErrorString(retval));
				} 
				else
				{
					printf("Decode returned!\n");
				}
				
			}
			else
			{
				printf("Unable to load audio into voice channel with error code %d (%s)\n", retval, port.ReturnErrorString(retval));
			}
		}
		else
		{
			printf("Unable to activate grammar with error code %d (%s)\n", retval, port.ReturnErrorString(retval));
		}
	}
	else
	{
		printf("Unable to load grammar with error code %d (%s)\n", retval, port.ReturnErrorString(retval));
	}

	//=========================================================================
	//collect results
	int num_parses, p_idx, num_interps, i_idx;

	num_parses = port.GetNumberOfNBestAlternatives(vc);

	if (num_parses > 0)
	{
		for (p_idx = 0; p_idx < num_parses; ++p_idx)
		{
			retval = port.SwitchToNBestAlternative(vc, p_idx);

			if (retval != LV_SUCCESS)
			{
				printf("Unable to use alternative %d of %\n", p_idx + 1, num_parses);

				exit_code = 8;

				break;
			}

			printf("Alternative %d:\n", p_idx + 1);

			num_interps = port.GetNumberOfInterpretations(vc);

			if (num_interps == 0)
			{
				printf("  No Interpretations for this alternate: %d\n", p_idx);
			}
			else
			{
				for (i_idx = 0; i_idx < num_interps; ++i_idx)
				{
					const char * input_string = port.GetInterpretationInputString(vc, i_idx);

					const char * gram_label = port.GetInterpretationGrammarLabel(vc, i_idx);

					const char * interp_string = port.GetInterpretationString(vc, i_idx);

					int score = port.GetInterpretationScore(vc, i_idx);

					printf("  Interpretation %d of %d\n", i_idx + 1, num_interps);
					printf("    Grammar Label        : %s\n", (gram_label?gram_label:""));
					printf("    Input Sentence       : %s\n", (input_string?input_string:""));
					printf("    Interpretation String: %s\n", (interp_string?interp_string:""));
					printf("    Interpretation Score : %d\n", score);
				}
				
			}
			
		}
		
	}

	port.DestroyClient();

	port.Shutdown();

#else

	HPORT port;

	port = LV_SRE_CreateClient(&OpenErrorCode, NULL, NULL, 0);
	
	if(port != NULL)
	{
		retval = LV_SRE_SetPropertyEx(port, PROP_EX_SAVE_SOUND_FILES, PROP_EX_VALUE_TYPE_INT, (void *)1, PROP_EX_TARGET_PORT, 0);

		if (retval != LV_SUCCESS)
		{
			printf("Unable to set property for PROP_EX_SAVE_SOUND_FILES with error code %d (%s)\n", retval, LV_SRE_ReturnErrorString(retval));
		}		
	
		retval = LV_SRE_SetPropertyEx(port, PROP_EX_MAX_NBEST_RETURNED, PROP_EX_VALUE_TYPE_INT, (void*)10, PROP_EX_TARGET_PORT, 0);

		if (retval != LV_SUCCESS)
		{
			printf("Unable to set property for PROP_EX_MAX_NBEST_RETURNED with error code %d (%s)\n", retval, LV_SRE_ReturnErrorString(retval));
		}		

		retval = LV_SRE_SetPropertyEx(port, PROP_EX_DECODE_TIMEOUT, PROP_EX_VALUE_TYPE_INT, (void*)100000, PROP_EX_TARGET_PORT, 0);

		if (retval != LV_SUCCESS)
		{
			printf("Unable to set property for PROP_EX_DECODE_TIMEOUT with error code %d (%s)\n", retval, LV_SRE_ReturnErrorString(retval));
		}		

		// 2. Load Grammar
		//LV_SRE_IsGrammarLoaded(port, grammar_label.c_str());
		retval = LV_SRE_LoadGrammar(port, grammar_label.c_str(), grammar_file.c_str());

		if(retval == LV_SUCCESS)
		{
			printf("Loaded Grammar!\n");

			// 3. Activate Grammar
			retval = LV_SRE_ActivateGrammar(port, grammar_label.c_str());

			if(retval == LV_SUCCESS)
			{
				// 4. Load Audio
				retval = LV_SRE_LoadVoiceChannel(port, vc, audio_buffer, flength, audio_format);

				if(retval == LV_SUCCESS)
				{
					printf("Loaded Audio!\n");

					// 5. Do Decode
					retval = LV_SRE_Decode(port, vc, LV_ACTIVE_GRAMMAR_SET, LV_DECODE_BLOCK | LV_DECODE_SEMANTIC_INTERPRETATION);

					if(retval < 0)
					{
						printf("Decode failed with error code %d (%s)\n", retval, LV_SRE_ReturnErrorString(retval));

						exit_code = 7;
					}
					else
					{
						printf("Decode returned!\n");
					}

				}
				else
				{
					printf("Unable to load audio into voice channel with error code %d (%s)\n", retval, LV_SRE_ReturnErrorString(retval));
				
					exit_code = 6;
				}

				LV_SRE_UnloadGrammars(port);
			}
			else	// Activate Grammar Failed
			{
				printf("Unable to activate the grammar with error code %d (%s)\n", retval, LV_SRE_ReturnErrorString(retval));

				exit_code = 5;
			}
		}
		else	// Load Grammar Failed
		{
			printf("Unable to load grammar with error code %d (%s)\n", retval, LV_SRE_ReturnErrorString(retval));

			exit_code = 4;
		}
	}
	else	// Open Port Failed
	{
		printf("Opening port failed with error code %d (%s)\n", OpenErrorCode, LV_SRE_ReturnErrorString(OpenErrorCode));

		exit_code = 3;
	}

	if(exit_code == 0)
	{
		// ==== Print Results ====
		// You may get multiple parses, and each parse may have multiple interpretations.
		int num_parses, p_idx, num_interps, i_idx;
	
		num_parses = LV_SRE_GetNumberOfNBestAlternatives(port,vc);

		if(num_parses > 0)
		{
			for(p_idx = 0; p_idx < num_parses; ++p_idx)
			{
				retval = LV_SRE_SwitchToNBestAlternative(port, vc, p_idx);

				if(retval != LV_SUCCESS)
				{
					printf("Unable to use alternative %d of %d\n", p_idx + 1, num_parses);

					exit_code = 8;

					break;
				}

				printf("Alternative %d:\n", p_idx + 1);

				num_interps = LV_SRE_GetNumberOfInterpretations(port, vc);

				if(num_interps == 0)
				{
					printf("  No. Interpretations for this alternate: %d\n", p_idx + 1);
				}
				else
				{
					for(i_idx = 0; i_idx < num_interps; ++i_idx)
					{
						const char * input_string = LV_SRE_GetInterpretationInputString(port, vc, i_idx);

						const char *gram_label = LV_SRE_GetInterpretationGrammarLabel(port, vc, i_idx);

						const char * interp_string = LV_SRE_GetInterpretationString(port, vc, i_idx);

						int score = LV_SRE_GetInterpretationScore(port, vc, i_idx);

						printf("  Interpretation %d of %d\n", i_idx + 1, num_interps);
						printf("    Grammar Label        : %s\n", (gram_label?gram_label:""));
						printf("    Input Sentence       : %s\n", (input_string?input_string:""));
						printf("    Interpretation String: %s\n", (interp_string?interp_string:""));
						printf("    Interpretation Score : %d\n", score);

					}
				}

			}
		}
		else
		{
			printf("No Decode results to report or there was a decode failure you might want to check the log files.\n");
		}
	}	
	
	// 6. Close the port, do not forget to do this.
	if(exit_code != 3)
	{
		LV_SRE_DestroyClient(port);
	}

	LV_SRE_Shutdown();// flush out any pending background operations like writing callsre files
#endif
	delete [] audio_buffer;

	return exit_code;
}

