/*
	
	******************************************

	text2file intended for Asterisk usage

	Execute the following command, in order to
	create 'verbiot2f' application:

	g++ vt2f.cpp -Wall -o verbiot2f -lvoxlib

	******************************************

*/

#include <iostream>
#include <time.h>
#include <getopt.h>
#include <fstream>
#include <cctype> // toupper tolower
#include <algorithm> // for transform

#include "voxlib.h"

//#define CODER  MC_ALAW
//#define EXT    ".esa"

using namespace std;

void show_help()
{
	cout << "***********************************************************************************************************************\n";
	cout << "* usage: vt2f -v -l <lang> -k <spkr> -t <text.txt> -f <audio_f> -c <codec> -d <devid> -s <vox_server_ip> \n";
	cout << "* - v: sets verbosity on                                                                                              " << endl;
	cout << "* - l: <lang>: es (spanish)                                                                                          " << endl;
	cout << "*               ca (catalan)                                                                                          " << endl;
	cout << "*               eu (Basque)                                                                                           " << endl;
	cout << "*               ga (Galician)                                                                                         " << endl;
	cout << "*               fr (French)                                                                                           " << endl;
	cout << "*               en-us (US-English)                                                                                   " << endl;
	cout << "*               es-mx (Mexican)                                                                                       " << endl;
	cout << "* - k: <spkr>:  Carlos    (Spanish male speaker)                                                                      " << endl;
	cout << "*               Laura     (Spanish female speaker)                                                                    " << endl;
	cout << "*               Pau       (Catalan male speaker)                                                                      " << endl;
	cout << "*               Meritxell (Catalan female speaker)                                                                    " << endl;
	cout << "*               Jon       (Basque male speaker)                                                                       " << endl;
	cout << "*               Freire    (Galician male speaker)                                                                     " << endl;
	cout << "*               Celia     (Mexican female speaker)                                                                    " << endl;
	cout << "* - t: <text.txt>: text we want to synth.                                                                             " << endl;
	cout << "* - f: <audio_f>: output file (alaw, ulaw or pcm).                                                                    " << endl;
	cout << "* - c: <codec>: audio format (of audio_file) LIN16(PCM) / ALAW / ULAW.                                          " << endl;
	cout << "* - d: <devid>: unique device id (channel identifier).                                                                " << endl;
	cout << "* - s: <vox_server_ip>: IP address of the voxserver we want to connect to.                                               " << endl;
	cout << "*********************************************************************************************************************** \n";
	cout << "" << endl;
}

void vox_error(const char *function, int dev)
{
	printf("*** ERROR: %s failed %s ***\n", function, ATVOX_ERRMSGP(dev));
}

int fatal_error( string msg)
{
	cout << "" << endl;
	cout << "FATAL ERROR: " << msg << endl;
	cout << "" << endl;
	show_help();
	return -1;	
}

/* sample source code */
int main( int argc, char *argv[])
{	
	//startup default options//
	string language 	= "es";
	string serverIp 	= "127.0.0.1";
	string speakerName 	= "laura";
	string devid		= "0";
	int dev 		= 0;
	long CODER 		= MC_MULAW; /*MC_LIN16, MC_ALAW, MC_MULAW*/
	string audio_codec	= "ULAW";
	string EXT		= "ulaw";
	bool verbose_on		= false;

	//synth options//
	string textToSynth 	= "verbio demo.";
	string textFileToSynth 	= "default.txt";
	string audioFileName 	= "default.wav";
	
	//Get command line options
	int c;
	int longopt_index = 0;
	extern int opterr;
	int show_usage = 0;
	char *optstring = "l:k:t:f:c:d:s:h:v";
	struct option long_options[] = {
		{ "help", 	0, 0, 'h' },
		{ "verbose", 	0, 0, 'v' },
		{ "language", 	0, 0, 'l' },
		{ "speaker", 	0, 0, 'k' },
		{ "text_file", 	0, 0, 't' },
		{ "file", 	0, 0, 'f' },
		{ "codec", 	0, 0, 'c' },
		{ "devid", 	0, 0, 'd' },
		{ "server", 	0, 0, 's' },
		{0, 0, 0, 0}
	};

	opterr = 0;

	while ((c = getopt_long (argc, argv, optstring, long_options, &longopt_index)) != -1) {
		switch (c) {
		case 1:
			/* getopt signals end of '-' options */
			break;

		case 'h':
			show_usage++;
			break;
		case 'v':
			verbose_on = true;
			break;
		case 'l':
			language = optarg;
			break;
		case 'k':
			speakerName = optarg;
			break;
		case 't':
			textFileToSynth = optarg;
			break;
		case 'f':
			audioFileName = optarg;
			break;
		case 'c':
			audio_codec = optarg;
			break;
		case 'd':
			devid = optarg;
			break;
		case 's':
			serverIp = optarg;
			break;
		default:
			fprintf (stderr, "error\n");
			show_usage++;
			break;
		}
	}
	
	if (show_usage) 
	{
		show_help();
		exit (1);
	}

	//
	//Prepare data
	//
	
	// Get data (text) from input file
	string line;
	ifstream myfile( textFileToSynth.c_str());
	if( myfile.is_open())
	{
		textToSynth = "";
		while( !myfile.eof() )
		{
			getline( myfile,line);
			textToSynth += " "; 
			textToSynth += line; 
		}
		myfile.close();
	}
	else
	{
		return fatal_error("ERROR: could not open " + textFileToSynth);
	}
	
	// Set devid
	dev = atoi(devid.c_str());

	// Langs and speakers to lower
	std::transform( language.begin(), 
				language.end(), 
				language.begin(), 
				(int(*)(int))tolower);

	std::transform( speakerName.begin(), 
				speakerName.end(), 
				speakerName.begin(), 
				(int(*)(int))tolower);

	//Set codec
	std::transform( audio_codec.begin(), 
				audio_codec.end(), 
				audio_codec.begin(), 
				(int(*)(int))toupper);
	if( audio_codec == "LIN16"   || 
		audio_codec == "WAV" || 
		audio_codec == "PCM")
	{
		CODER = MC_LIN16;
		EXT = "pcm";
	}
	else if(audio_codec == "WAV")
	{
		CODER = MC_LIN16;
		EXT = "pcm";
	}
	else if(audio_codec == "ALAW")
	{
		CODER = MC_ALAW;
		EXT = "alaw";
	}
	else if(audio_codec == "ULAW")
	{
		CODER = MC_MULAW;
		EXT = "ulaw";
	}
	else
		return fatal_error("Invalid audio codec: " + audio_codec);


	//***************************************************//
	//*************** 0.Show params *********************//
	//***************************************************//
	if( verbose_on)
	{
		cout << " " << endl;
		cout << "Configuration (Execute vt2f -h for help):" << endl;
		cout << " - Voxserver IP: " 	<< serverIp << endl;
		cout << " - Language: " 	<< language << endl;
		cout << " - speakerName: " 	<< speakerName << endl;
		cout << " - Text to synth: \"" 	<< textToSynth << "\"" << endl;
		cout << "             " 	<< endl;
		cout << " - CODER: \"" 		<< audio_codec << "\"" << endl;
		cout << " - Output file: " 	<< audioFileName << endl;
		cout << "----------" 	<< endl;	
	}

	//***************************************************//
	//***************** 1.startup ***********************//
	//***************************************************//
	// Set server ip 
	if( vox_setparm( dev, VXGB_DEFSERVER, serverIp.c_str()) == -1)
	{
		vox_error("vox_setparm", dev);
		return ATVOX_LASTERR(-1);
	}
	
	//if( vox_libinit( OVX_NORECSTR | language.c_str()  /* | OVX_TTS16K */) == -1) 
	if( vox_tts_init( NULL, language.c_str()) == -1) 
	{
		vox_error( "vox_libinit", dev);
		return ATVOX_LASTERR( -1);
	}


	//***************************************************//
	//****************** 2.synthesis ********************//
	//***************************************************//
	if (!vox_setparm(dev, VXCH_DEFTTSLANG, language.c_str()) && 
		!vox_setparm(dev,VXCH_TTSSPKNAME, speakerName.c_str()))
	{
		//vox_getparm(dev, VXCH_TTSSPKNAME, &spkname);
		//strcpy(filename, spkname); strcat(filename, EXT);
		string full_name = audioFileName + "." + EXT;
		if( vox_playstr(dev, full_name.c_str(), textToSynth.c_str(), CODER) == -1)
		{
			printf("Error on vox_playstr %s\n", speakerName.c_str());
			vox_error( "vox_playstr", dev);
		}
	}
	else
	{
		vox_error( "vox_setparm", dev);
		return fatal_error( "ERROR: vox_setparm synth");
	}
	
// 	// Default Speaker
// 	if( !vox_playstr(dev, audioFileName.c_str(), text, CODER))
// 		printf("FILENAME = %s\n", filename);
// 	else
// 		vox_error("vox_playstr", dev);

	vox_libclose();

	return 1;
}
